-- G P S  script for Rising World
-- Displays world position and heading of the player according to common navigation practices
--
-- waypoints.lua: manages waypoints (including Home)
--
-- (C) Maurizio M. Gavioli (a.k.a. Miwarre), 2016
-- Licensed under the Creative Commons by-sa 3.0 license (see http://creativecommons.org/licenses/by-sa/3.0/ for details)

-- listWp(player)
--
-- Lists all defined waypoints (including home) for player

function listWp(player)
	local	waypoints = player:getAttribute(key_gpsWaypoints)
	player:sendTextMessage(msg_wpList)
	for i=0, 9 do
		wp = waypoints[i]
		if wp ~= nil and wp.name ~= nil and wp.name ~= "" then
			player:sendTextMessage(i..": "..wp.name.." ("..math.floor(wp.z + 0.5).."N,"..-math.floor(wp.x + 0.5).."E) h"..math.floor(wp.y + 0.5) )
		end
	end
end

-- setWp(player, par1, par2)
--
-- Sets a new waypoint for player according to par1 and par2.
-- par1 and par2 contain the index and the name of the waypoint to create, in either order;
-- either or both can be null, in which case an index and/or a name are supplied by the function;
-- but if both are present they cannot be both indices or both names.
-- If an index is not given, the function reuses the first available slot, if any, or overwrite
-- waypoint 1 if no lto is available

function setWp(player, par1, par2)
	local idx	= 0									-- prepare empty waypoint index and name
	local name	= ""
	local idx0

	if par1 ~= nil then								-- at least 1 parameter
		if tonumber(par1) ~= nil then				-- check par 1 is a number
			idx		= tonumber(par1)				-- use par 2 as an index
			if math.floor(idx) ~= idx then			-- check it is an integer
				player:sendMessage(err_setWpInvalidIndex)
				return
			end
		else
			name	= par1							-- ... otherwise use par 1 as a name
		end

		if par2 ~= nil then							-- 2 parameters
			if tonumber(par2) ~= nil then			-- check par 2 is a number...
				if idx ~= 0 then					-- ... if we already a index => error
					player:sendTextMessage(err_setWpDuplIdx)
					return
				end
				idx		= tonumber(par2)			-- use par 2 as an index
				if math.floor(idx) ~= idx then		-- check it is an integer
					player:sendMessage(err_setWpInvalidIndex)
					return
				end
			else
				if name ~= "" then					-- ... if we already have a name => error
					player:sendTextMessage(err_setWpDuplName)
					return
				end
				name	= par2						-- ... use par 2 as a name
			end
		end
	end
	-- if no idx, look for a suitable index
	if idx == 0 then
		for i = 1,9 do
			wp	= player:getAttribute(key_gpsWaypoints)[i]
			if  wp == nil or wp.name == nil or wp.name == "" then
				idx = i
				break
			end
		end
	end
	if idx == 0 then	idx = 1		end			-- if still no index, use first index
	-- if idx out of range => error
	if idx < 0 or idx > 9 then
		player:sendTextMessage(err_setWpIdxOutOfRange)
		return
	end
	-- if no name, create a name from wp index
	if name == "" then
		name = string.format("wp%d", idx)
	end
	dbSetWp(player, idx, name)
end

-- setShowWp(player, index)
--
-- turn on/off display of a waypoint data. Index = 0 => turn off waypoint data display

function setShowWp(player, index)
	index = toWpIndex(player, index, err_showWpInvalidIndex)
	if not index then
		return
	end
	-- if not turning off (index = 0), check that wp exists
	if index > 0 and player:getAttribute(key_gpsWaypoints)[index] == nil then
		player:sendTextMessage(string.format(err_showWpUndefinedWp, index))
		return
	end
	player:setAttribute(key_gpsWpGUI, index)
	setGPSText(player)						-- update displayed text
end

-- teleportToWp(player, index)
--
-- teleports to the index-th wp (incl. home)

function teleportToWp(player, index)
	index = toWpIndex(player, index, err_showWpInvalidIndex)
	if not index then
		return
	end
	-- check teleporting to wp is enabled
	if index > 0 and not allowTpToWaypoints then
		player:sendTextMessage(err_noTpToWp)
		return
	end
	-- check that wp exists
	local wp = player:getAttribute(key_gpsWaypoints)[index]
	if wp == nil then
		player:sendTextMessage(string.format(err_showWpUndefinedWp, index))
		return
	end
	player:setPosition(wp.x, wp.y, wp.z)
	setGPSText(player)						-- update displayed text
end

-- toWpIndex(index)
--
-- converts the given index to a valid wp index (0-9) if possible
-- on failure displays a msg and return false

function toWpIndex(player, index, err_msg)
	if index == nil or tonumber(index) == nil then		-- check index is supplied and is a number
		return false;										-- set to invalid index
	end
	index = tonumber(index)
	if math.floor(index) ~= index then					-- check index is an integer
		return false
	end
	if index < 0 or index > 9 then						-- index out of range
		player:sendTextMessage(err_msg)
		return false
	end
	return index
end
