-- G P S  script for Rising World
-- Displays world position and heading of the player according to common navigation practices
--
-- (C) Maurizio M. Gavioli (a.k.a. Miwarre), 2016
-- Licensed under the Creative Commons by-sa 3.0 license (see http://creativecommons.org/licenses/by-sa/3.0/ for details)

--------------------
-- G L O B A L S
--------------------
-- Settings
allowTpToWaypoints	= false			-- whether teleporting to waypoints (in addition to home) is possible or not
nameDispLen			= 8				-- the max length of waypoint names to display on screen
rad2deg				= 57.295779513	-- 180 / PI
wpHdgPrecis			= 5				-- the waypoint radial delta below which route corrections arrows are not displayed

database			= getDatabase()
gpsVersion			= 0.1

-- attribute keys
key_gpsInfoGUI		= "gpsInfoGUI"
key_gpsHomeGUI		= "gpsHomeGUI"
key_gpsInfoLabel	= "gpsInfoLabel"
key_gpsWaypoints	= "gpsWaypoints"
key_gpsWpGUI		= "gpsWpGUI"

-- INCLUDE - after globals are defined
include("messages.lua")				-- include before any other include
include("database.lua")
include("waypoints.lua")

--------------------
-- E V E N T S
--------------------

-- onEnable()
--
-- called at script loading. Required for the script to work.
 
function onEnable()
	dbInit()							-- init DB, if required
	print(msg_init)
end

-- onPlayerConnect
--
-- called when the player connects to the world.
-- Creates and set the GPS text and bool up as attributes of the player

function onPlayerConnect(event)
	-- GUI
	local gpsInfoLabel = Gui:createLabel("")		-- the text to show on screen
	gpsInfoLabel:setX(0.50)
	gpsInfoLabel:setY(0.10)
	gpsInfoLabel:setFontColor(0xFFFFFFFF)
	gpsInfoLabel:setBackgroundColor(0x0000007f)
	gpsInfoLabel:setFontsize(18)
	gpsInfoLabel:setPivot(4)
	event.player:setAttribute(key_gpsInfoLabel, gpsInfoLabel)
	event.player:addGuiElement(gpsInfoLabel)

	event.player:setAttribute(key_gpsInfoGUI, true)		-- whether the GPS text is shown or not
	event.player:setAttribute(key_gpsHomeGUI, false)	-- whether the home info is shown or not
	event.player:setAttribute(key_gpsWpGUI, 0)			-- which waypoint is shown, if any (0 = none)

	-- DB
	dbLoadPlayer(event.player)
	setGPSText(event.player)							-- set ibitially displayed text
end
addEvent("PlayerConnect", onPlayerConnect);

-- onPlayerCommand
--
-- called when the player issues a command ("/...") in the chat window
-- Toggles the GPS info display on the "/gps" command

function onPlayerCommand(event)
	local command = event.command
	local cmd = StringUtils:explode(command, " ")

	if cmd[1] == "/gps" then
	local player	= event.player
		-- if no sub-command, flip text display on/off
		if #cmd == 1 then
			setGPSShow(player, not player:getAttribute(key_gpsInfoGUI))
			return
		end

		if #cmd == 2 then
			if cmd[2] == "on" then
				setGPSShow(player, true)
				return
			end
			if cmd[2] == "off" then
				setGPSShow(player, false)
				return
			end
			if cmd[2] == "help" then
				help(player)
				return
			end
			if cmd[2] == "showhome" then
				player:setAttribute(key_gpsHomeGUI, not player:getAttribute(key_gpsHomeGUI) )
				setGPSText(player)				-- update displayed text
				return
			end
			if cmd[2] == "list" then
				listWp(player)
				return
			end
			if cmd[2] == "sethome" then
				dbSetHome(player)
				return
			end
			if cmd[2] == "home" then
				teleportToWp(player, 0)
				return
			end
		end

		if #cmd == 3 then
			if cmd[2] == "showwp" then
				setShowWp(player, cmd[3])
				return
			end
			if cmd[2] == "goto" then
				teleportToWp(player, cmd[3])
				return
			end
		end

		-- "/gps setwp" has a variable number of parameters
		if cmd[2] == "setwp" then
			setWp(player, cmd[3], cmd[4])
			return
		end

		player:sendTextMessage(err_gspInvalidCmd..command.."'")	-- if the command has not understood => error
	end

	-- 2 commands added for compatibility with other common scripts
	if cmd[1] == "/sethome" then
		dbSetHome(player)
	end
	if cmd[1] == "/home" then
		teleportToWp(player, 0)
	end
end
addEvent("PlayerCommand", onPlayerCommand);

-- onPlayerChangePosition
--
-- called when the player changes of world position.
-- Note: change of view direction without any actual displacement
-- does not always trigger this event, particularly while flying

function onPlayerChangePosition(event)
	setGPSText(event.player);
end
addEvent("PlayerChangePosition", onPlayerChangePosition);

--------------------
-- U T I L I T Y  F U N C T I O N S
--------------------

-- setGPSText(player)
--
-- actually fills the player GPS info text

function setGPSText(player)
	local labelgpsInfo = player:getAttribute(key_gpsInfoLabel);
	if labelgpsInfo == nil then
		return;
	end

	if player:getAttribute(key_gpsInfoGUI) then
		local text = ""
		-- PLAYER ROTATION
		local playerRot	= player:getViewDirection();
		-- x- and z-components of the viewing vector
		local rotX		= playerRot.x;
		local rotZ		= playerRot.z;
		-- if the viewing vector is not horizontal but slanted up or down,
		-- the x and z components are shortened: scale them up until
		-- they are the catheti of a triangle whose hypotenuse is long 1
		local scale		= math.sqrt(rotX*rotX + rotZ*rotZ);
		-- get the heading angle and convert to degrees
		local heading	= math.acos(rotZ / scale) * rad2deg;
		-- hdg is correct from N (0°) through E (90°) to S (180°)
		-- then, when view is toward W, it decreases down to 0° again;
		-- when view is toward W, correct by taking the complementary angle
		if rotX > 0		then heading = 360 - heading;	end
		-- round to nearest integer and uniform 0° to 360°
		heading	= math.floor(heading + 0.5);
		if heading == 0	then heading = 360;				end

		-- PLAYER POSITION
		local playerPos = player:getPosition();
		local posE		= math.floor(-playerPos.x + 0.5);	-- convert positive W to standard, positive E
		local posN		= math.floor(playerPos.z + 0.5);
		local posH		= math.floor(playerPos.y + 0.5);

		-- OUTPUT
		-- home
		local wp		= player:getAttribute(key_gpsWaypoints)[0]
		if player:getAttribute(key_gpsHomeGUI) and wp ~= nil then
			text = wpText(wp, heading, playerPos).." | "
		end
		-- main data
		text			= text .. string.format("%03d°", heading).." ("..posN.."N,"..posE.."E) h"..posH
		-- waypoint
		local wpToShow	= player:getAttribute(key_gpsWpGUI)
		if wpToShow > 0 then
			wp			= player:getAttribute(key_gpsWaypoints)[wpToShow]
			if wp ~= nil then
				text = text .. " | " .. wpText(wp, heading, playerPos)
			end
		end

		labelgpsInfo:setText(text);
	else
		labelgpsInfo:setText("");
	end
end

-- wpText(wp, playerHdg, playerPos)
--
-- returns the text to display for the given waypoint with respect to the given player heading and position

function wpText(wp, playerHdg, playerPos)
	local deltaN	= wp.z - playerPos.z
	local deltaW	= wp.x - playerPos.x
	local dist		= math.sqrt(deltaN * deltaN + deltaW * deltaW)			-- distance in blocks
	local radial
	if dist < 4 then			-- if distance less than 2 m, data are unreliable, only output wp name
		return  " | ---°  "..string.sub(wp.name, 1, nameDispLen).."   <2m"
	else
		radial	= math.acos(deltaN / dist) * rad2deg
		if deltaW > 0	then radial = 360 - radial;	end		-- for this adjustment,  see setGPSText() above
		radial			= math.floor(radial + 0.5);
		if radial == 0	then radial = 360;				end
	end
--	-- text build up
	local wpHdgDelta= playerHdg - radial
	local text		= string.format("%03d°", radial)						-- separator and radial
	if (wpHdgDelta > wpHdgPrecis and wpHdgDelta < (180-wpHdgPrecis))		-- left arrow
			or (wpHdgDelta > (wpHdgPrecis-360) and wpHdgDelta < (-wpHdgPrecis-180)) then
		text = text .. " <"
	else
		text = text .. "  "
	end
	text  = text .. string.sub(wp.name, 1, nameDispLen)						-- wp name
	if (wpHdgDelta < -wpHdgPrecis and wpHdgDelta > (wpHdgPrecis-180))		-- right arrow
			or (wpHdgDelta < (360-wpHdgPrecis) and wpHdgDelta > (wpHdgPrecis+180)) then
		text = text .. "> "
	else
		text = text .. "  "
	end
	text = text .. math.floor(dist / 2 + 0.5) .. "m"						-- distance in m
	return text
end

-- setGPSShow(player, show)
--
-- sets whether the GPS text is shown or not for player

function setGPSShow(player, show)
	player:setAttribute(key_gpsInfoGUI, show)
	setGPSText(player)						-- update displayed text
end

-- help()
--
-- Displays help summary

function help(player)
	for k,v in pairs(txt_help) do
		player:sendTextMessage(v)
	end
end
