-- T E L e v a t o r 
-- Implements elevators via teleporting
--
-- levels.lua: manages elevator levels
--
-- (C) Maurizio M. Gavioli (a.k.a. Miwarre), 2016
-- Licensed under the Creative Commons by-sa 3.0 license (see http://creativecommons.org/licenses/by-sa/3.0/ for details)

--------------------
-- L E V E L S
--------------------

-- toElevatorLevel(player, levelOrd)
--
-- Checks if the player is in an elevator level and moves him to the requested level of that elevator.
-- Returns 0 on success; -1 on syntax error; -2 on player not in an elevator; -3 on no such a level;
--		-4 on player already on that level

function toElevatorLevel(player, levelOrd)
	levelOrd = toInteger(levelOrd)
	if levelOrd == nil then
		return -1						-- syntax error
	end
	local playerPos = player:getPosition()
	local level = getLevelAtPos(playerPos.x, playerPos.y, playerPos.z)
	if level == nil then
		return -2						-- not in an elevator
	end
	if level.level == levelOrd then
		return -4						-- already at that level
	end
	level = getLevelAtLevel(level.elevId, levelOrd)
	if level == nil then
		return -3						-- no such a destination level
	else
		-- teleport to the middle of the destination level platform, a bit too high
		player:setPosition((level.globlXto + level.globlXfr) / 2, level.landH + 0.05, (level.globlZto + level.globlZfr) / 2)
	end
	return 0
end

--
-- getLevelAtPos(x, y, z)
--
-- returns the level existing at the given position or nil if none if found

function getLevelAtPos(x, y, z)
	for key,value in pairs(levels) do
		if x >= value["globlXfr"] and x <= value["globlXto"] and
				y >= value["globlYfr"] and y <= value["globlYto"] and
				z >= value["globlZfr"] and z <= value["globlZto"] then
			return value
		end
	end
	return nil
end

--
-- getLevelAtLevel(elevId, levelOrd)
--
-- returns the idx-th level of the elevator with given id or nil if none if found

function getLevelAtLevel(elevId, levelOrd)
	for key,value in pairs(levels) do
		if value["elevId"] == elevId and value["level"] == levelOrd then
			return value
		end
	end
	return nil
end

--
-- normalizeBoundaries(area)
--
-- Swaps From and To boundaries for each From > To

function normaliseBoundaries(area)
	if ChunkUtils:getGlobalBlockPositionX(area["chunkXfr"], area["blockXfr"]) >
			ChunkUtils:getGlobalBlockPositionX(area["chunkXto"], area["blockXto"]) then
		area["chunkXfr"], area["chunkXto"] = area["chunkXto"], area["chunkXfr"]
		area["blockXfr"], area["blockXto"] = area["blockXto"], area["blockXfr"]
	end
	if ChunkUtils:getGlobalBlockPositionY(area["chunkYfr"], area["blockYfr"]) >
			ChunkUtils:getGlobalBlockPositionY(area["chunkYto"], area["blockYto"]) then
		area["chunkYfr"], area["chunkYto"] = area["chunkYto"], area["chunkYfr"]
		area["blockYfr"], area["blockYto"] = area["blockYto"], area["blockYfr"]
	end
	if ChunkUtils:getGlobalBlockPositionZ(area["chunkZfr"], area["blockZfr"]) >
			ChunkUtils:getGlobalBlockPositionZ(area["chunkZto"], area["blockZto"]) then
		area["chunkZfr"], area["chunkZto"] = area["chunkZto"], area["chunkZfr"]
		area["blockZfr"], area["blockZto"] = area["blockZto"], area["blockZfr"]
	end
end

--
-- getGlobalPos(area)
--
-- Computes the "global" start and end position of an area from chunk and block positions.

function getGlobalPos(area)
	area["globlXfr"] = ChunkUtils:getGlobalBlockPositionX(area["chunkXfr"], area["blockXfr"]);
	area["globlYfr"] = ChunkUtils:getGlobalBlockPositionY(area["chunkYfr"], area["blockYfr"]);
	area["globlZfr"] = ChunkUtils:getGlobalBlockPositionZ(area["chunkZfr"], area["blockZfr"]);
	area["globlXto"] = ChunkUtils:getGlobalBlockPositionX(area["chunkXto"], area["blockXto"]);
	area["globlYto"] = ChunkUtils:getGlobalBlockPositionY(area["chunkYto"], area["blockYto"]);
	area["globlZto"] = ChunkUtils:getGlobalBlockPositionZ(area["chunkZto"], area["blockZto"]);
end

--------------------
-- E L E V A T O R S
--------------------

--
-- getElevator
--
-- Returns the elevator with the given ID or nil if it does not exists

function getElevator(elevId)
	return elevators[elevId]
end
