
package com.example.rw.lock;

import net.risingworld.api.Plugin;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.player.PlayerCommandEvent;
import net.risingworld.api.events.player.PlayerStorageAccessEvent;
import net.risingworld.api.events.player.world.PlayerChangeObjectStatusEvent;
import net.risingworld.api.events.player.PlayerObjectInteractionEvent;
import net.risingworld.api.objects.Player;
import net.risingworld.api.objects.Storage;
import net.risingworld.api.database.Database;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Locale;

public final class ChestDoorLockPlugin extends Plugin {

    private Database db;

    @Override
    public void onEnable() {
        try {
            db = getSQLiteConnection(getPath() + "/lockdb.db");
            db.execute("CREATE TABLE IF NOT EXISTS locks (" +
                    "object_type TEXT NOT NULL, " +       // 'storage' | 'door'
                    "object_id   INTEGER NOT NULL PRIMARY KEY, " +
                    "owner_uid   TEXT NOT NULL, " +
                    "is_public   INTEGER NOT NULL DEFAULT 0, " + // 0/1
                    "created_at  INTEGER NOT NULL" +
                    ");");
            db.execute("CREATE TABLE IF NOT EXISTS lock_allow (" +
                    "object_id  INTEGER NOT NULL, " +
                    "player_uid TEXT NOT NULL, " +
                    "UNIQUE(object_id, player_uid) ON CONFLICT IGNORE" +
                    ");");
            registerEventListener(this);
            getLogger().log("ChestDoorLockPlugin enabled.");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onDisable() {
        try { if (db != null) db.close(); } catch (Exception ignored) {}
    }

    // ===== Core helpers =====

    private boolean isLocked(String type, long id) {
        try (ResultSet rs = db.executeQuery("SELECT 1 FROM locks WHERE object_type='" + type + "' AND object_id=" + id + " LIMIT 1;")) {
            return rs.next();
        } catch (SQLException e) { e.printStackTrace(); return false; }
    }

    private String getOwnerUID(String type, long id) {
        try (ResultSet rs = db.executeQuery("SELECT owner_uid FROM locks WHERE object_type='" + type + "' AND object_id=" + id + " LIMIT 1;")) {
            if (rs.next()) return rs.getString(1);
        } catch (SQLException e) { e.printStackTrace(); }
        return null;
    }

    private boolean isPublic(String type, long id) {
        try (ResultSet rs = db.executeQuery("SELECT is_public FROM locks WHERE object_type='" + type + "' AND object_id=" + id + " LIMIT 1;")) {
            if (rs.next()) return rs.getInt(1) == 1;
        } catch (SQLException e) { e.printStackTrace(); }
        return false;
    }

    private boolean isAllowed(long id, String playerUID) {
        try (ResultSet rs = db.executeQuery("SELECT 1 FROM lock_allow WHERE object_id=" + id + " AND player_uid='" + playerUID + "' LIMIT 1;")) {
            return rs.next();
        } catch (SQLException e) { e.printStackTrace(); return false; }
    }

    private void setLock(String type, long id, String ownerUID) {
        db.executeUpdate("INSERT OR REPLACE INTO locks(object_type, object_id, owner_uid, is_public, created_at) " +
                "VALUES('" + type + "', " + id + ", '" + ownerUID + "', 0, " + (System.currentTimeMillis()/1000L) + ");");
    }

    private void removeLock(long id) {
        db.executeUpdate("DELETE FROM locks WHERE object_id=" + id + ";");
        db.executeUpdate("DELETE FROM lock_allow WHERE object_id=" + id + ";");
    }

    private void addAllow(long id, String uid) {
        db.executeUpdate("INSERT OR IGNORE INTO lock_allow(object_id, player_uid) VALUES(" + id + ", '" + uid + "');");
    }

    private void removeAllow(long id, String uid) {
        db.executeUpdate("DELETE FROM lock_allow WHERE object_id=" + id + " AND player_uid='" + uid + "';");
    }

    private void setPublic(long id, boolean pub) {
        db.executeUpdate("UPDATE locks SET is_public=" + (pub ? 1 : 0) + " WHERE object_id=" + id + ";");
    }

    // ===== Command handling =====
    @EventMethod
    public void onCommand(PlayerCommandEvent evt) {
        String[] args = evt.getCommand().trim().split("\\s+");
        if (args.length == 0) return;
        if (!args[0].equalsIgnoreCase("/lock")) return;

        evt.setCancelled(true);
        Player p = evt.getPlayer();

        if (args.length == 1) {
            p.sendTextMessage("[Lock] Commands: /lock set | allow <name> | deny <name> | public on|off | info | remove");
            return;
        }

        String sub = args[1].toLowerCase(Locale.ROOT);
        switch (sub) {
            case "set" -> handleSet(p);
            case "allow" -> {
                if (args.length < 3) { p.sendTextMessage("[Lock] Usage: /lock allow <player>"); return; }
                handleAllow(p, args[2], true);
            }
            case "deny" -> {
                if (args.length < 3) { p.sendTextMessage("[Lock] Usage: /lock deny <player>"); return; }
                handleAllow(p, args[2], false);
            }
            case "public" -> {
                if (args.length < 3 || !(args[2].equalsIgnoreCase("on") || args[2].equalsIgnoreCase("off"))) {
                    p.sendTextMessage("[Lock] Usage: /lock public on|off");
                    return;
                }
                handlePublic(p, args[2].equalsIgnoreCase("on"));
            }
            case "info" -> handleInfo(p);
            case "remove" -> handleRemove(p);
            default -> p.sendTextMessage("[Lock] Unknown subcommand.");
        }
    }

    private record Target(String type, long id, String label) {}

    private Target findLookTarget(Player p) {
        Storage nearest = p.getNearestStorage(4f);
        if (nearest != null && !nearest.isTransient()) {
            return new Target("storage", nearest.getID(), "Chest " + nearest.getID());
        }
        var hit = p.getLookAtObject(4f);
        if (hit != null && hit.getDefinition() != null) {
            var def = hit.getDefinition();
            if (def.isDoor()) {
                long gid = hit.getGlobalID();
                return new Target("door", gid, "Door " + gid);
            }
        }
        return null;
    }

    private void handleSet(Player p) {
        Target t = findLookTarget(p);
        if (t == null) { p.sendTextMessage("[Lock] Look at a chest or a door within ~4m."); return; }
        setLock(t.type, t.id, p.getUID());
        p.sendTextMessage("[Lock] Locked " + t.label + " (owner: you).");
    }

    private void handleAllow(Player p, String playerName, boolean add) {
        Target t = findLookTarget(p);
        if (t == null) { p.sendTextMessage("[Lock] Look at a chest or a door within ~4m."); return; }
        String owner = getOwnerUID(t.type, t.id);
        if (owner == null) { p.sendTextMessage("[Lock] This target isn’t locked yet. Use /lock set first."); return; }
        if (!owner.equals(p.getUID()) && !p.isAdmin()) { p.sendTextMessage("[Lock] Only owner or admin may change ACL."); return; }

        Player other = getServer().getPlayerByName(playerName);
        if (other == null) { p.sendTextMessage("[Lock] Player not found (they must be online)."); return; }

        if (add) { addAllow(t.id, other.getUID()); p.sendTextMessage("[Lock] Allowed " + other.getName()); }
        else { removeAllow(t.id, other.getUID()); p.sendTextMessage("[Lock] Removed " + other.getName()); }
    }

    private void handlePublic(Player p, boolean pub) {
        Target t = findLookTarget(p);
        if (t == null) { p.sendTextMessage("[Lock] Look at a chest or a door within ~4m."); return; }
        String owner = getOwnerUID(t.type, t.id);
        if (owner == null) { p.sendTextMessage("[Lock] This target isn’t locked yet. Use /lock set first."); return; }
        if (!owner.equals(p.getUID()) && !p.isAdmin()) { p.sendTextMessage("[Lock] Only owner or admin may change ACL."); return; }
        setPublic(t.id, pub);
        p.sendTextMessage("[Lock] Public access " + (pub ? "enabled" : "disabled") + ".");
    }

    private void handleInfo(Player p) {
        Target t = findLookTarget(p);
        if (t == null) { p.sendTextMessage("[Lock] Look at a chest or a door within ~4m."); return; }
        if (!isLocked(t.type, t.id)) { p.sendTextMessage("[Lock] Not locked."); return; }
        String owner = getOwnerUID(t.type, t.id);
        boolean pub = isPublic(t.type, t.id);
        p.sendTextMessage("[Lock] " + t.label + " — owner: " + owner + ", public: " + pub);
    }

    private void handleRemove(Player p) {
        Target t = findLookTarget(p);
        if (t == null) { p.sendTextMessage("[Lock] Look at a chest or a door within ~4m."); return; }
        String owner = getOwnerUID(t.type, t.id);
        if (owner == null) { p.sendTextMessage("[Lock] Not locked."); return; }
        if (!owner.equals(p.getUID()) && !p.isAdmin()) { p.sendTextMessage("[Lock] Only owner/admin may remove."); return; }
        removeLock(t.id);
        p.sendTextMessage("[Lock] Lock removed.");
    }

    // ===== Enforcement =====

    @EventMethod
    public void onStorageAccess(PlayerStorageAccessEvent evt) {
        Storage s = evt.getStorage();
        if (s == null || s.isTransient()) return;
        long id = s.getID();
        if (!isLocked("storage", id)) return;
        Player p = evt.getPlayer();

        if (p.isAdmin()) return;
        if (getOwnerUID("storage", id).equals(p.getUID())) return;
        if (isPublic("storage", id)) return;
        if (isAllowed(id, p.getUID())) return;

        evt.setCancelled(true);
        p.sendTextMessage("[Lock] This chest is locked.");
    }

    @EventMethod
    public void onDoorStatusChange(PlayerChangeObjectStatusEvent evt) {
        var def = evt.getObjectDefinition();
        if (def == null || !def.isDoor()) return;
        long gid = evt.getGlobalID();
        if (!isLocked("door", gid)) return;
        Player p = evt.getPlayer();

        if (p.isAdmin()) return;
        if (getOwnerUID("door", gid).equals(p.getUID())) return;
        if (isPublic("door", gid)) return;
        if (isAllowed(gid, p.getUID())) return;

        evt.setCancelled(true);
        p.sendTextMessage("[Lock] This door is locked.");
    }

    @EventMethod
    public void onDoorInteract(PlayerObjectInteractionEvent evt) {
        var def = evt.getObjectDefinition();
        if (def == null || !def.isDoor()) return;
        long gid = evt.getGlobalID();
        if (!isLocked("door", gid)) return;
        Player p = evt.getPlayer();

        if (p.isAdmin()) return;
        if (getOwnerUID("door", gid).equals(p.getUID())) return;
        if (isPublic("door", gid)) return;
        if (isAllowed(gid, p.getUID())) return;

        evt.setCancelled(true);
        p.sendTextMessage("[Lock] This door is locked.");
    }
}
