/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.DiscordService;
import com.example.risingworld.discordadmin.PluginLogger;
import java.awt.Color;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import net.dv8tion.jda.api.EmbedBuilder;
import net.dv8tion.jda.api.entities.Message;
import net.dv8tion.jda.api.entities.MessageEmbed;
import net.dv8tion.jda.api.entities.MessageType;
import net.dv8tion.jda.api.entities.channel.concrete.TextChannel;
import net.dv8tion.jda.api.exceptions.ErrorResponseException;
import net.risingworld.api.Server;
import net.risingworld.api.objects.Player;

public class StatusManager {
    private final DiscordService discordService;
    private final PluginLogger logger;
    private final Properties config;
    private final AtomicReference<Message> statusMessage = new AtomicReference<Object>(null);
    private ScheduledExecutorService scheduler;
    private ScheduledFuture<?> statusUpdateTask;
    private volatile boolean isRestarting = false;
    private volatile boolean isInitialized = false;
    private volatile boolean isInitializing = false;
    private final Object initLock = new Object();
    private final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private Color onlineColor;
    private Color offlineColor;
    private Color restartColor;
    private int maxPlayersToList;
    private boolean showPlayerTime;
    private boolean showAdmin;
    private boolean recreateOnStartup;
    private final Map<String, Long> playerConnectionTimes = new HashMap<String, Long>();

    public StatusManager(DiscordService discordService, PluginLogger logger, Properties config) {
        this.discordService = discordService;
        this.logger = logger;
        this.config = config;
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.loadConfiguration();
        if (discordService.isConnected() && discordService.getAdminChannel() != null) {
            this.initialize();
        }
    }

    public void trackPlayerConnect(String playerName) {
        this.playerConnectionTimes.put(playerName.toLowerCase(), System.currentTimeMillis());
    }

    public void removePlayerTracking(String playerName) {
        this.playerConnectionTimes.remove(playerName.toLowerCase());
    }

    private void loadConfiguration() {
        if (this.config != null) {
            String onlineColorStr = this.config.getProperty("status.embed.online.color", "#00FF00");
            String offlineColorStr = this.config.getProperty("status.embed.offline.color", "#FF0000");
            String restartColorStr = this.config.getProperty("status.embed.restart.color", "#FFA500");
            this.onlineColor = this.parseColor(onlineColorStr);
            this.offlineColor = this.parseColor(offlineColorStr);
            this.restartColor = this.parseColor(restartColorStr);
            this.maxPlayersToList = Integer.parseInt(this.config.getProperty("status.embed.max_players", "15"));
            this.showPlayerTime = Boolean.parseBoolean(this.config.getProperty("status.embed.show_player_time", "false"));
            this.showAdmin = Boolean.parseBoolean(this.config.getProperty("status.embed.show_admin", "false"));
            this.recreateOnStartup = Boolean.parseBoolean(this.config.getProperty("status.embed.recreate_on_startup", "false"));
            this.logger.debug("Status configuration: maxPlayersToList=" + this.maxPlayersToList + ", showPlayerTime=" + this.showPlayerTime + ", showAdmin=" + this.showAdmin + ", recreateOnStartup=" + this.recreateOnStartup);
        } else {
            this.onlineColor = new Color(0, 255, 0);
            this.offlineColor = new Color(255, 0, 0);
            this.restartColor = new Color(255, 165, 0);
            this.maxPlayersToList = 15;
            this.showPlayerTime = false;
            this.showAdmin = false;
            this.recreateOnStartup = false;
        }
    }

    private Color parseColor(String colorStr) {
        try {
            if (colorStr.startsWith("#")) {
                return Color.decode(colorStr);
            }
            return (Color)Color.class.getField(colorStr.toUpperCase()).get(null);
        }
        catch (Exception e) {
            this.logger.warning("Invalid color format: " + colorStr + ", using default");
            return Color.GRAY;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initialize() {
        TextChannel channel = this.discordService.getAdminChannel();
        if (channel == null) {
            this.logger.warning("Cannot initialize server status - Discord channel is null");
            return;
        }
        Object object = this.initLock;
        synchronized (object) {
            if (this.isInitializing) {
                this.logger.debug("Status initialization already in progress, skipping duplicate request");
                return;
            }
            this.isInitializing = true;
        }
        ScheduledFuture[] timeoutTask = new ScheduledFuture[]{this.scheduler.schedule(() -> {
            Object object = this.initLock;
            synchronized (object) {
                if (this.isInitializing) {
                    this.logger.warning("Status initialization timed out after 30 seconds");
                    this.isInitializing = false;
                }
            }
        }, 30L, TimeUnit.SECONDS)};
        channel.retrievePinnedMessages().queue(pinnedMessages -> {
            try {
                Message existingStatus = this.findExistingStatusMessage((List<Message>)pinnedMessages);
                if (existingStatus != null) {
                    this.statusMessage.set(existingStatus);
                    this.logger.debug("Found existing server status message");
                    this.updateStatusMessage(true);
                    this.isInitialized = true;
                } else {
                    this.createNewStatusMessage();
                }
            }
            finally {
                Object object = this.initLock;
                synchronized (object) {
                    this.isInitializing = false;
                }
                if (timeoutTask[0] != null) {
                    timeoutTask[0].cancel(false);
                }
            }
        }, error -> {
            this.logger.log(Level.WARNING, "Failed to retrieve pinned messages: " + error.getMessage(), (Throwable)error);
            Object object = this.initLock;
            synchronized (object) {
                this.isInitializing = false;
            }
            if (timeoutTask[0] != null) {
                timeoutTask[0].cancel(false);
            }
            this.scheduleRetryInitialization(5000L);
        });
    }

    private void scheduleRetryInitialization(long delayMillis) {
        this.scheduler.schedule(() -> this.createNewStatusMessage(), delayMillis, TimeUnit.MILLISECONDS);
    }

    private Message findExistingStatusMessage(List<Message> pinnedMessages) {
        if (pinnedMessages.isEmpty()) {
            return null;
        }
        for (Message pinned : pinnedMessages) {
            List embeds;
            if (!pinned.getAuthor().getId().equals(this.discordService.getJDA().getSelfUser().getId()) || (embeds = pinned.getEmbeds()).isEmpty() || !"Server Status".equals(((MessageEmbed)embeds.get(0)).getTitle())) continue;
            if (this.recreateOnStartup) {
                pinned.delete().queue(success -> this.logger.debug("Deleted previous server status message"), error -> this.logger.warning("Failed to delete previous status message: " + error.getMessage()));
                try {
                    Thread.sleep(500L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                return null;
            }
            this.logger.debug("Using existing server status message");
            return pinned;
        }
        return null;
    }

    private void createNewStatusMessage() {
        try {
            this.logger.debug("Creating new server status message");
            MessageEmbed embed = this.createStatusEmbed(true, "", false);
            this.discordService.getAdminChannel().sendMessageEmbeds(embed, new MessageEmbed[0]).queue(message -> {
                this.statusMessage.set((Message)message);
                TextChannel channel = this.discordService.getAdminChannel();
                message.pin().queue(success -> {
                    this.logger.debug("Server status message pinned successfully");
                    this.isInitialized = true;
                    this.scheduler.schedule(() -> channel.getHistory().retrievePast(10).queue(messages -> {
                        for (Message msg : messages) {
                            if (msg.getType() != MessageType.CHANNEL_PINNED_ADD || !msg.getTimeCreated().toInstant().isAfter(message.getTimeCreated().toInstant()) || msg.getReferencedMessage() == null || !msg.getReferencedMessage().getId().equals(message.getId())) continue;
                            msg.delete().queue(deleteSuccess -> this.logger.debug("Deleted pin notification message"), deleteError -> this.logger.fine("Failed to delete pin notification: " + deleteError.getMessage()));
                            break;
                        }
                    }, error -> this.logger.fine("Failed to fetch messages for pin notification cleanup: " + error.getMessage())), 12L, TimeUnit.SECONDS);
                }, error -> {
                    this.logger.warning("Failed to pin server status message: " + error.getMessage());
                    this.isInitialized = true;
                });
            }, error -> {
                this.logger.log(Level.WARNING, "Failed to send server status message: " + error.getMessage(), (Throwable)error);
                this.isInitializing = false;
                this.scheduleRetryInitialization(10000L);
            });
        }
        finally {
            this.isInitializing = false;
        }
    }

    public ScheduledFuture<?> scheduleStatusUpdates(int intervalMinutes) {
        this.logger.debug("Scheduling status updates every " + intervalMinutes + " minutes");
        if (intervalMinutes < 1) {
            intervalMinutes = 10;
            this.logger.warning("Invalid status update interval, using default of 10 minutes");
        }
        if (this.statusUpdateTask != null && !this.statusUpdateTask.isDone()) {
            this.statusUpdateTask.cancel(false);
        }
        this.statusUpdateTask = this.scheduler.scheduleAtFixedRate(() -> {
            try {
                if (this.isInitialized) {
                    this.updatePlayerStatus();
                } else if (!this.isInitializing) {
                    this.initialize();
                }
            }
            catch (Exception e) {
                this.logger.log(Level.WARNING, "Error in scheduled status update", e);
            }
        }, 1L, intervalMinutes, TimeUnit.MINUTES);
        return this.statusUpdateTask;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateStatusMessage(boolean online, String customStatus, boolean isRestarting) {
        ReentrantLock statusUpdateLock = new ReentrantLock();
        try {
            if (!statusUpdateLock.tryLock(5L, TimeUnit.SECONDS)) {
                this.logger.warning("Could not acquire status update lock after 5 seconds - update aborted");
                return;
            }
            try {
                Message message = this.statusMessage.get();
                if (message == null) {
                    this.logger.debug("Status message not initialized yet, initializing now");
                    statusUpdateLock.unlock();
                    this.initialize();
                    return;
                }
                this.isRestarting = isRestarting;
                MessageEmbed embed = this.createStatusEmbed(online, customStatus, isRestarting);
                CompletableFuture updateFuture = new CompletableFuture();
                message.editMessageEmbeds(new MessageEmbed[]{embed}).queue(success -> {
                    this.logger.debug("Server status message updated successfully");
                    updateFuture.complete(null);
                }, error -> {
                    ErrorResponseException ere;
                    this.logger.warning("Failed to update server status message: " + error.getMessage());
                    if (error instanceof ErrorResponseException && ((ere = (ErrorResponseException)error).getErrorCode() == 10008 || ere.getErrorCode() == 404)) {
                        this.logger.debug("Status message appears to have been deleted, reinitializing");
                        this.statusMessage.set(null);
                        this.scheduleRetryInitialization(5000L);
                    }
                    updateFuture.completeExceptionally((Throwable)error);
                });
                try {
                    updateFuture.get(10L, TimeUnit.SECONDS);
                }
                catch (TimeoutException e) {
                    this.logger.warning("Status update timed out after 10 seconds");
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            finally {
                statusUpdateLock.unlock();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.warning("Thread interrupted while waiting for status update lock");
        }
    }

    public void updateStatusMessage(boolean online) {
        this.updateStatusMessage(online, null, this.isRestarting);
    }

    private String formatPlayerConnectionTime(String playerName) {
        Long connectTime = this.playerConnectionTimes.get(playerName.toLowerCase());
        if (connectTime == null) {
            connectTime = System.currentTimeMillis();
            this.playerConnectionTimes.put(playerName.toLowerCase(), connectTime);
            return "just connected";
        }
        long elapsedTime = System.currentTimeMillis() - connectTime;
        return this.formatDuration(elapsedTime);
    }

    private String formatDuration(long milliseconds) {
        long seconds = milliseconds / 1000L;
        long minutes = seconds / 60L;
        long hours = minutes / 60L;
        if (hours > 0L) {
            return hours + "h " + minutes % 60L + "m";
        }
        if (minutes > 0L) {
            return minutes + "m";
        }
        return seconds + "s";
    }

    private MessageEmbed createStatusEmbed(boolean online, String customStatus, boolean isRestarting) {
        EmbedBuilder builder = new EmbedBuilder();
        builder.setTitle("Server Status");
        if (online) {
            if (isRestarting) {
                builder.setColor(this.restartColor);
                builder.setDescription((CharSequence)"\ud83d\udd04 Server is **RESTARTING**");
                builder.addField("Status", customStatus != null ? customStatus : "Restart Scheduled", false);
            } else {
                float uptime;
                builder.setColor(this.onlineColor);
                builder.setDescription((CharSequence)"\u2705 Server is **ONLINE**");
                int playerCount = Server.getPlayerCount();
                int maxPlayers = this.getMaxPlayers();
                builder.addField("Players Online", playerCount + "/" + maxPlayers, false);
                if (playerCount > 0) {
                    try {
                        ArrayList<String> playerNames = new ArrayList<String>();
                        ArrayList<String> playerInfos = new ArrayList<String>();
                        for (Player player : Server.getAllPlayers()) {
                            String playerName = player.getName();
                            playerNames.add(playerName);
                            StringBuilder playerInfo = new StringBuilder(playerName);
                            if (this.showAdmin && player.isAdmin()) {
                                playerInfo.append(" [Admin]");
                            }
                            if (this.showPlayerTime) {
                                String connectTime = this.formatPlayerConnectionTime(playerName);
                                playerInfo.append(" (").append(connectTime).append(")");
                            }
                            playerInfos.add(playerInfo.toString());
                        }
                        String playerList = this.formatPlayerList(playerInfos);
                        builder.addField("Player List", playerList, false);
                    }
                    catch (Exception e) {
                        this.logger.log(Level.WARNING, "Error while getting player information", e);
                        builder.addField("Player List", "Error retrieving player information", false);
                    }
                }
                if ((uptime = Server.getRunningTime()) > 0.0f) {
                    long hours = (long)uptime / 3600L;
                    long minutes = (long)uptime % 3600L / 60L;
                    builder.addField("Uptime", hours + "h " + minutes + "m", true);
                }
            }
        } else {
            builder.setColor(this.offlineColor);
            builder.setDescription((CharSequence)"\u274c Server is **OFFLINE**");
        }
        String timestamp = this.dateFormat.format(new Date());
        builder.setFooter("Last updated: " + timestamp);
        return builder.build();
    }

    private String formatPlayerList(List<String> playerInfos) {
        if (playerInfos.isEmpty()) {
            return "No players online";
        }
        Object playerList = String.join((CharSequence)"\n", playerInfos.subList(0, Math.min(playerInfos.size(), this.maxPlayersToList)));
        if (playerInfos.size() > this.maxPlayersToList) {
            playerList = (String)playerList + "\nAnd " + (playerInfos.size() - this.maxPlayersToList) + " more players...";
        }
        return playerList;
    }

    private int getMaxPlayers() {
        try {
            return Integer.parseInt(Server.getOption("Settings_MaxPlayers"));
        }
        catch (NumberFormatException e) {
            return 32;
        }
    }

    public void updatePlayerStatus() {
        if (!this.isInitialized && !this.isInitializing) {
            this.initialize();
            return;
        }
        this.updateStatusMessage(true);
    }

    public void setServerOffline() {
        if (this.statusMessage.get() == null) {
            this.logger.debug("Status message is null during setServerOffline");
            return;
        }
        this.updateStatusMessage(false, "Server is offline", false);
    }

    public void setServerOnline() {
        this.isRestarting = false;
        this.updateStatusMessage(true);
    }

    public void setServerRestarting(int minutesRemaining) {
        this.isRestarting = true;
        this.updateStatusMessage(true, "Restart in " + minutesRemaining + " minutes", true);
    }

    public boolean isInitialized() {
        return this.isInitialized;
    }

    public void shutdown() {
        try {
            Message message;
            if (this.statusUpdateTask != null && !this.statusUpdateTask.isDone()) {
                this.statusUpdateTask.cancel(false);
            }
            if (this.scheduler != null) {
                this.scheduler.shutdown();
                if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                }
            }
            if ((message = this.statusMessage.get()) != null) {
                this.setServerOffline();
            }
            this.isInitialized = false;
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error during status manager shutdown", e);
        }
    }
}

