/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.FileUtils;
import com.example.risingworld.discordadmin.PluginLogger;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

public class LibraryLoader {
    private final PluginLogger logger;
    private final String pluginPath;
    private final AtomicBoolean downloadInProgress = new AtomicBoolean(false);
    private final ExecutorService downloadExecutor = Executors.newFixedThreadPool(3);
    private final ScheduledExecutorService scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
    private Runnable dependencyCallback = null;
    private static final Map<String, DependencyInfo> DEPENDENCIES = new HashMap<String, DependencyInfo>();
    private static final String MAVEN_CENTRAL = "https://repo1.maven.org/maven2/";
    private static final String MAVEN_METADATA_XML = "maven-metadata.xml";

    public LibraryLoader(PluginLogger logger, String pluginPath) {
        this.logger = logger;
        this.pluginPath = pluginPath;
    }

    public void setDependencyCallback(Runnable callback) {
        this.dependencyCallback = callback;
    }

    public boolean initialize() {
        this.logger.debug("Starting library loader initialization...");
        try {
            boolean apiAvailable = this.checkRisingWorldApiAvailable();
            if (!apiAvailable) {
                this.logger.warning("Rising World API not found in classpath. Scanning common locations...");
                boolean apiLoaded = this.loadRisingWorldApiFromCommonLocations();
                if (!apiLoaded) {
                    this.logger.severe("Could not find or load Rising World API. Plugin cannot function without this dependency.");
                    return false;
                }
            }
            String report = this.generateEnvironmentReport();
            this.logger.debug("Environment Report generated");
            try {
                File reportFile = new File(this.pluginPath, "environment_report.txt");
                try (FileOutputStream fos = new FileOutputStream(reportFile);){
                    fos.write(report.getBytes());
                }
                this.logger.debug("Saved environment report to: " + reportFile.getAbsolutePath());
            }
            catch (Exception e) {
                this.logger.warning("Failed to save environment report: " + e.getMessage());
            }
            if (this.checkDependencies()) {
                this.logger.info("All required dependencies are already available!");
                return true;
            }
            List<String> missingDependencies = this.checkMissingDependencies();
            if (!missingDependencies.isEmpty()) {
                this.logger.info("Missing dependencies detected: " + String.join((CharSequence)", ", missingDependencies) + " - Attempting download in background");
                Thread downloadThread = new Thread(() -> {
                    try {
                        boolean downloadSuccess = this.downloadMissingDependencies(missingDependencies);
                        if (downloadSuccess) {
                            this.logger.info("Successfully downloaded all missing dependencies!");
                            this.loadLibraries();
                            if (this.checkDependencies()) {
                                this.logger.info("All required dependencies now available. Plugin can be initialized.");
                                if (this.dependencyCallback != null) {
                                    this.dependencyCallback.run();
                                }
                            } else {
                                this.logger.warning("Some dependencies still unavailable after download. Please check logs.");
                            }
                        } else {
                            this.logger.warning("Failed to download all required dependencies automatically.");
                            this.logger.warning("Please manually download the required libraries to: " + new File(this.pluginPath, "lib").getAbsolutePath());
                        }
                    }
                    catch (Exception e) {
                        this.logger.log(Level.SEVERE, "Error during dependency download", e);
                    }
                });
                downloadThread.setName("DependencyDownloader");
                downloadThread.setDaemon(true);
                downloadThread.start();
            }
            this.logger.debug("Plugin initialization deferred until all dependencies are available.");
            return false;
        }
        catch (Exception e) {
            this.logger.severe("Error during library loader initialization: " + e.getMessage());
            return false;
        }
    }

    private void startBackgroundDependencyChecker() {
        this.logger.debug("Starting background dependency checker (checks every 30 seconds)");
        String report = this.generateEnvironmentReport();
        this.logger.debug("Environment Report generated for background dependency checker");
        try {
            File reportFile = new File(this.pluginPath, "environment_report.txt");
            try (FileOutputStream fos = new FileOutputStream(reportFile);){
                fos.write(report.getBytes());
            }
            this.logger.debug("Saved environment report to: " + reportFile.getAbsolutePath());
        }
        catch (Exception e) {
            this.logger.warning("Failed to save environment report: " + e.getMessage());
        }
        this.logger.info("Server will continue running, but some functionality may be limited.");
        this.logger.info("Please add missing libraries to: " + new File(this.pluginPath, "lib").getAbsolutePath());
        this.scheduledExecutor.scheduleAtFixedRate(() -> {
            try {
                this.logger.debug("Checking if dependencies are now available...");
                if (this.loadLibraries()) {
                    this.logger.info("All dependencies now available! Full functionality restored.");
                    if (this.dependencyCallback != null) {
                        this.dependencyCallback.run();
                    }
                    this.scheduledExecutor.shutdown();
                }
            }
            catch (Exception e) {
                this.logger.debug("Error checking dependencies in background: " + e.getMessage());
            }
        }, 30L, 30L, TimeUnit.SECONDS);
    }

    public void shutdown() {
        try {
            this.downloadExecutor.shutdown();
            this.scheduledExecutor.shutdown();
            if (!this.downloadExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.downloadExecutor.shutdownNow();
            }
            if (!this.scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduledExecutor.shutdownNow();
            }
            this.logger.debug("Library loader shutdown complete");
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.warning("Library loader shutdown interrupted");
        }
    }

    public boolean loadLibraries() {
        try {
            List<String> missingDependencies;
            File[] libFiles;
            File libFolder = new File(this.pluginPath + "/lib");
            if (!libFolder.exists()) {
                libFolder.mkdirs();
                this.logger.debug("Created lib directory at " + libFolder.getAbsolutePath());
            }
            if ((libFiles = libFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar"))) != null) {
                for (File libFile : libFiles) {
                    try {
                        this.loadJarFile(libFile);
                        this.logger.debug("Loaded library: " + libFile.getName());
                    }
                    catch (Exception e) {
                        this.logger.warning("Failed to load library: " + libFile.getName() + " - " + e.getMessage());
                    }
                }
            }
            if (!(missingDependencies = this.checkMissingDependencies()).isEmpty()) {
                this.logger.debug("Detected missing dependencies. Attempting to download automatically...");
                boolean downloadSuccess = this.downloadMissingDependencies(missingDependencies);
                if (downloadSuccess) {
                    libFiles = libFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar"));
                    if (libFiles != null) {
                        for (File libFile : libFiles) {
                            try {
                                this.loadJarFile(libFile);
                                this.logger.debug("Loaded library: " + libFile.getName());
                            }
                            catch (Exception e) {
                                this.logger.warning("Failed to load library: " + libFile.getName() + " - " + e.getMessage());
                            }
                        }
                    }
                    if ((missingDependencies = this.checkMissingDependencies()).isEmpty()) {
                        this.logger.info("All required dependencies have been successfully downloaded and loaded");
                        return true;
                    }
                    this.logger.warning("Still missing some dependencies after download attempt: " + String.join((CharSequence)", ", missingDependencies));
                    return false;
                }
                this.logger.warning("Failed to download all required dependencies automatically");
                this.logger.warning("Server will continue running, but with limited functionality.");
                this.logger.warning("Please manually download the required libraries to: " + libFolder.getAbsolutePath());
                return false;
            }
            return missingDependencies.isEmpty();
        }
        catch (Exception e) {
            this.logger.severe("Error loading libraries: " + e.getMessage());
            return false;
        }
    }

    private boolean checkRisingWorldApiAvailable() {
        try {
            Class.forName("net.risingworld.api.Plugin");
            Class.forName("net.risingworld.api.Server");
            return true;
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    private boolean loadRisingWorldApiFromCommonLocations() {
        File parentRoot;
        File pluginDir;
        File file;
        ArrayList<Object> possibleLocations = new ArrayList<Object>();
        possibleLocations.add("./api");
        possibleLocations.add("../api");
        possibleLocations.add("../../api");
        String os = System.getProperty("os.name").toLowerCase();
        if (os.contains("windows")) {
            possibleLocations.add("C:/Program Files/Rising World/api");
            possibleLocations.add("C:/Program Files (x86)/Rising World/api");
            possibleLocations.add("C:/Program Files/Steam/steamapps/common/Rising World/api");
            possibleLocations.add("C:/Program Files (x86)/Steam/steamapps/common/Rising World/api");
        } else if (os.contains("mac")) {
            possibleLocations.add(System.getProperty("user.home") + "/Library/Application Support/Steam/steamapps/common/Rising World/api");
        } else {
            possibleLocations.add(System.getProperty("user.home") + "/.steam/steam/steamapps/common/Rising World/api");
            possibleLocations.add("/opt/risingworld/api");
            possibleLocations.add("/opt/rising_world/api");
        }
        possibleLocations.add("../RisingWorld/api");
        possibleLocations.add("../../RisingWorld/api");
        possibleLocations.add("../../../RisingWorld/api");
        if (this.pluginPath != null && (file = (pluginDir = new File(this.pluginPath)).getParentFile()) != null && file.exists() && (parentRoot = file.getParentFile()) != null && parentRoot.exists()) {
            possibleLocations.add(new File(parentRoot, "api").getAbsolutePath());
            possibleLocations.add(new File(parentRoot, "lib").getAbsolutePath());
        }
        for (String string : possibleLocations) {
            File[] jarFiles;
            File apiDir = new File(string);
            this.logger.debug("Checking for Rising World API in: " + apiDir.getAbsolutePath());
            if (!apiDir.exists() || !apiDir.isDirectory() || (jarFiles = apiDir.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar") && (name.toLowerCase().contains("api") || name.toLowerCase().contains("rising")))) == null || jarFiles.length <= 0) continue;
            for (File jar : jarFiles) {
                try {
                    this.loadJarFile(jar);
                    this.logger.debug("Loaded potential Rising World API from: " + jar.getAbsolutePath());
                    if (!this.checkRisingWorldApiAvailable()) continue;
                    this.logger.info("Successfully loaded Rising World API from: " + jar.getAbsolutePath());
                    return true;
                }
                catch (Exception e) {
                    this.logger.debug("Failed to load jar as Rising World API: " + jar.getAbsolutePath());
                }
            }
        }
        this.logger.severe("Could not find Rising World API in any common location");
        return false;
    }

    private void loadJarFile(File jarFile) throws Exception {
        if (jarFile == null || !jarFile.exists() || !jarFile.isFile()) {
            throw new IllegalArgumentException("Invalid JAR file: " + (jarFile == null ? "null" : jarFile.getAbsolutePath()));
        }
        try {
            ClassLoader classLoader = this.getClass().getClassLoader();
            try {
                if (classLoader instanceof URLClassLoader) {
                    Method addUrlMethod = URLClassLoader.class.getDeclaredMethod("addURL", URL.class);
                    addUrlMethod.setAccessible(true);
                    addUrlMethod.invoke((Object)classLoader, jarFile.toURI().toURL());
                    this.logger.debug("Loaded JAR via URLClassLoader: " + jarFile.getName());
                    return;
                }
            }
            catch (Exception urlClassLoaderEx) {
                this.logger.debug("Could not use URLClassLoader method: " + urlClassLoaderEx.getMessage());
            }
            try {
                Method moduleAddUrlMethod = classLoader.getClass().getDeclaredMethod("addURL", URL.class);
                moduleAddUrlMethod.setAccessible(true);
                moduleAddUrlMethod.invoke((Object)classLoader, jarFile.toURI().toURL());
                this.logger.debug("Loaded JAR via module-aware classloader: " + jarFile.getName());
                return;
            }
            catch (Exception moduleLoaderEx) {
                this.logger.debug("Could not use module-aware classloader method: " + moduleLoaderEx.getMessage());
                try {
                    Method systemClassPathMethod = ClassLoader.getSystemClassLoader().getClass().getDeclaredMethod("appendToClassPathForInstrumentation", String.class);
                    systemClassPathMethod.setAccessible(true);
                    systemClassPathMethod.invoke((Object)ClassLoader.getSystemClassLoader(), jarFile.getAbsolutePath());
                    this.logger.debug("Loaded JAR via system classloader: " + jarFile.getName());
                }
                catch (Exception systemLoaderEx) {
                    this.logger.warning("Failed to load JAR file using any available method: " + jarFile.getName());
                    throw new RuntimeException("Unable to load JAR file", systemLoaderEx);
                }
            }
        }
        catch (Exception e) {
            this.logger.warning("Comprehensive error loading JAR file " + jarFile.getName() + ": " + e.getMessage());
            throw e;
        }
    }

    private List<String> checkMissingDependencies() {
        try {
            File[] libFiles;
            File libFolder = new File(this.pluginPath + "/lib");
            if (libFolder.exists() && (libFiles = libFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar"))) != null && libFiles.length > 0) {
                for (File libFile : libFiles) {
                    try {
                        this.loadJarFile(libFile);
                        this.logger.debug("Loaded library during dependency check: " + libFile.getName());
                    }
                    catch (Exception e) {
                        this.logger.debug("Failed to load library during dependency check: " + libFile.getName() + " - " + e.getMessage());
                    }
                }
            }
        }
        catch (Exception e) {
            this.logger.debug("Error loading library files during dependency check: " + e.getMessage());
        }
        ArrayList<String> missingDependencies = new ArrayList<String>();
        for (Map.Entry<String, DependencyInfo> entry : DEPENDENCIES.entrySet()) {
            String dependencyName = entry.getKey();
            DependencyInfo info = entry.getValue();
            if (!info.isRequired()) continue;
            boolean available = this.isDependencyAvailable(info.getTestClass());
            if (!available) {
                missingDependencies.add(dependencyName);
                continue;
            }
            this.logger.debug("Found required dependency: " + dependencyName);
        }
        if (!missingDependencies.isEmpty()) {
            this.logger.debug("Missing dependencies after loading attempt: " + String.join((CharSequence)", ", missingDependencies));
        } else {
            this.logger.debug("All required dependencies are available");
        }
        return missingDependencies;
    }

    private boolean isDependencyAvailable(String className) {
        File[] libFiles;
        this.logger.debug("Checking dependency availability for: " + className);
        if (className == null || className.isEmpty()) {
            this.logger.debug("Dependency class name is null or empty");
            return false;
        }
        File libDirectory = new File(this.pluginPath + "/lib");
        if (libDirectory.exists() && (libFiles = libDirectory.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar"))) != null) {
            this.logger.debug("Loaded JAR files in lib directory:");
            for (File jar : libFiles) {
                this.logger.debug("- " + jar.getName());
            }
        }
        try {
            Class.forName(className);
            this.logger.debug("Successfully loaded class: " + className);
            return true;
        }
        catch (ClassNotFoundException e) {
            String[] alternativeClasses;
            this.logger.debug("Could not load class: " + className);
            for (String altClass : alternativeClasses = new String[0]) {
                try {
                    Class.forName(altClass);
                    this.logger.debug("Found alternative SLF4J class: " + altClass);
                    return true;
                }
                catch (ClassNotFoundException ignored) {
                    this.logger.debug("Alternative class not found: " + altClass);
                }
            }
            String classpath = System.getProperty("java.class.path");
            this.logger.debug("Current Java Classpath: " + classpath);
            this.logger.debug("Classloader Hierarchy:");
            for (ClassLoader currentLoader = this.getClass().getClassLoader(); currentLoader != null; currentLoader = currentLoader.getParent()) {
                this.logger.debug("- " + currentLoader.getClass().getName());
            }
            try {
                String artifactId = "slf4j-simple";
                File searchLibDirectory = new File(this.pluginPath + "/lib");
                if (searchLibDirectory.exists()) {
                    File[] matchingJars = searchLibDirectory.listFiles((dir, name) -> name.toLowerCase().contains(artifactId.toLowerCase()) && name.toLowerCase().endsWith(".jar"));
                    if (matchingJars != null && matchingJars.length > 0) {
                        for (File jar : matchingJars) {
                            this.logger.debug("Found potential SLF4J Simple JAR: " + jar.getName());
                            try {
                                this.loadJarFile(jar);
                                for (String altClass : alternativeClasses) {
                                    try {
                                        Class.forName(altClass);
                                        this.logger.debug("Successfully loaded alternative class after JAR load: " + altClass);
                                        return true;
                                    }
                                    catch (ClassNotFoundException altIgnored) {
                                        this.logger.debug("Still could not load: " + altClass);
                                    }
                                }
                            }
                            catch (Exception loadEx) {
                                this.logger.debug("Failed to load JAR: " + jar.getName() + " - " + loadEx.getMessage());
                            }
                        }
                    } else {
                        this.logger.debug("No JAR files found matching SLF4J Simple");
                    }
                }
            }
            catch (Exception ex) {
                this.logger.debug("Error in comprehensive SLF4J detection: " + ex.getMessage());
            }
            return false;
        }
    }

    private boolean containsMatchingDependencyName(String jarName, String className) {
        jarName = jarName.toLowerCase();
        String dependencyPart = "";
        if (className.contains(".")) {
            String[] parts;
            for (String part : parts = className.split("\\.")) {
                if (part.length() <= 2 || this.isCommonPackageName(part)) continue;
                if (jarName.contains(part.toLowerCase())) {
                    return true;
                }
                dependencyPart = part.toLowerCase();
            }
        }
        for (Map.Entry<String, DependencyInfo> entry : DEPENDENCIES.entrySet()) {
            String artifactId;
            String[] coords;
            DependencyInfo info = entry.getValue();
            String mavenCoords = info.getMavenCoordinates().toLowerCase();
            if (!info.getTestClass().equals(className) || (coords = mavenCoords.split(":")).length != 2 || !jarName.contains(artifactId = coords[1])) continue;
            return true;
        }
        return false;
    }

    private boolean isCommonPackageName(String name) {
        String[] commonNames;
        for (String common : commonNames = new String[]{"com", "org", "net", "io", "java", "javax", "api", "impl", "internal", "util", "utils", "common", "core"}) {
            if (!common.equalsIgnoreCase(name)) continue;
            return true;
        }
        return false;
    }

    public boolean checkDependencies() {
        List<String> missingDependencies = this.checkMissingDependencies();
        if (missingDependencies.isEmpty()) {
            this.logger.debug("All required dependencies are available");
            return true;
        }
        this.logger.debug("Missing dependencies: " + String.join((CharSequence)", ", missingDependencies));
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean downloadMissingDependencies(List<String> missingDependencies) {
        if (missingDependencies.isEmpty()) {
            return true;
        }
        this.logger.info("Attempting to download missing dependencies: " + String.join((CharSequence)", ", missingDependencies));
        if (!this.downloadInProgress.compareAndSet(false, true)) {
            this.logger.warning("Download already in progress, cannot start a new one");
            return false;
        }
        try {
            File libDir = new File(this.pluginPath + "/lib");
            if (!libDir.exists()) {
                libDir.mkdirs();
                this.logger.debug("Created lib directory at " + libDir.getAbsolutePath());
            }
            ConcurrentHashMap<String, Future<Boolean>> downloadFutures = new ConcurrentHashMap<String, Future<Boolean>>();
            for (String dependencyName : missingDependencies) {
                DependencyInfo info = DEPENDENCIES.get(dependencyName);
                if (info != null) {
                    String[] coordinates = info.getMavenCoordinates().split(":");
                    if (coordinates.length != 2) {
                        this.logger.warning("Invalid Maven coordinates format for: " + dependencyName);
                        continue;
                    }
                    String groupId = coordinates[0];
                    String artifactId = coordinates[1];
                    String version = this.resolveLatestVersion(groupId, artifactId, dependencyName);
                    if (version == null) {
                        this.logger.warning("Failed to resolve latest version for: " + dependencyName);
                        if (!"Trove4J".equals(dependencyName)) continue;
                        version = "3.1.0";
                    }
                    info.setResolvedVersion(version);
                    String repositoryPath = groupId.replace('.', '/') + "/" + artifactId + "/" + version + "/";
                    String filename = artifactId + "-" + version + ".jar";
                    String url = MAVEN_CENTRAL + repositoryPath + filename;
                    File outputFile = new File(libDir, filename);
                    Future<Boolean> future = this.downloadExecutor.submit(() -> this.downloadFile(url, outputFile));
                    downloadFutures.put(dependencyName, future);
                    this.logger.debug("Started download for: " + dependencyName + " (version " + version + ") from " + url);
                    continue;
                }
                this.logger.warning("No download information for dependency: " + dependencyName);
            }
            boolean allSuccessful = true;
            for (Map.Entry entry : downloadFutures.entrySet()) {
                String dependencyName = (String)entry.getKey();
                Future future = (Future)entry.getValue();
                try {
                    boolean success = (Boolean)future.get(60L, TimeUnit.SECONDS);
                    if (success) {
                        this.logger.info("Successfully downloaded: " + dependencyName + " (version " + DEPENDENCIES.get(dependencyName).getResolvedVersion() + ")");
                        continue;
                    }
                    this.logger.warning("Failed to download: " + dependencyName);
                    allSuccessful = false;
                }
                catch (Exception e) {
                    this.logger.warning("Error downloading " + dependencyName + ": " + e.getMessage());
                    future.cancel(true);
                    allSuccessful = false;
                }
            }
            boolean bl = allSuccessful;
            return bl;
        }
        catch (Exception e) {
            this.logger.severe("Error in dependency download process: " + e.getMessage());
            boolean bl = false;
            return bl;
        }
        finally {
            this.downloadInProgress.set(false);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String resolveLatestVersion(String groupId, String artifactId, String dependencyName) {
        try {
            if ("core".equals(artifactId) && "net.sf.trove4j".equals(groupId)) {
                return "3.1.0";
            }
            if ("okhttp".equals(artifactId) && "com.squareup.okhttp3".equals(groupId)) {
                return "4.12.0";
            }
            if ("jackson-databind".equals(artifactId) && "com.fasterxml.jackson.core".equals(groupId)) {
                return "2.17.2";
            }
            if ("jackson-core".equals(artifactId) && "com.fasterxml.jackson.core".equals(groupId)) {
                return "2.17.2";
            }
            if ("jackson-annotations".equals(artifactId) && "com.fasterxml.jackson.core".equals(groupId)) {
                return "2.17.2";
            }
            String metadataUrl = MAVEN_CENTRAL + groupId.replace('.', '/') + "/" + artifactId + "/maven-metadata.xml";
            this.logger.debug("Resolving latest version from: " + metadataUrl);
            URL url = new URI(metadataUrl).toURL();
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestProperty("User-Agent", "Discord-Admin-Plugin/3.4.1");
            connection.setConnectTimeout(10000);
            connection.setReadTimeout(20000);
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                this.logger.debug("Failed to access Maven metadata for " + dependencyName + ": HTTP " + responseCode);
                if (!"core".equals(artifactId)) return null;
                if (!"net.sf.trove4j".equals(groupId)) return null;
                return "3.1.0";
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));){
                String string;
                String[] versionTags;
                String versionsContent;
                String line;
                StringBuilder content = new StringBuilder();
                while ((line = reader.readLine()) != null) {
                    content.append(line);
                }
                String version = this.extractXmlTagContent(content.toString(), "release");
                if ((version == null || version.isEmpty()) && (versionsContent = this.extractXmlTagContent(content.toString(), "versions")) != null && (versionTags = versionsContent.split("<version>")).length > 1) {
                    version = versionTags[versionTags.length - 1].split("</version>")[0];
                }
                if (version == null || version.isEmpty()) {
                    this.logger.debug("Could not extract version for " + dependencyName);
                    if ("core".equals(artifactId) && "net.sf.trove4j".equals(groupId)) {
                        string = "3.1.0";
                        return string;
                    }
                    string = null;
                    return string;
                }
                this.logger.debug("Resolved latest version for " + dependencyName + ": " + version);
                string = version;
                return string;
            }
        }
        catch (Exception e) {
            this.logger.debug("Error resolving latest version for " + dependencyName + ": " + e.getMessage());
            if (!"core".equals(artifactId)) return null;
            if (!"net.sf.trove4j".equals(groupId)) return null;
            return "3.1.0";
        }
    }

    private String extractXmlTagContent(String xml, String tagName) {
        String openTag = "<" + tagName + ">";
        String closeTag = "</" + tagName + ">";
        int startIndex = xml.indexOf(openTag);
        if (startIndex == -1) {
            return null;
        }
        int endIndex = xml.indexOf(closeTag, startIndex += openTag.length());
        if (endIndex == -1) {
            return null;
        }
        return xml.substring(startIndex, endIndex).trim();
    }

    private boolean downloadFile(String urlString, File outputFile) {
        try {
            URL url;
            try {
                url = new URI(urlString).toURL();
            }
            catch (URISyntaxException e) {
                this.logger.debug("Invalid URI syntax: " + e.getMessage() + ", falling back to URL constructor");
                url = new URL(urlString);
            }
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestProperty("User-Agent", "Discord-Admin-Plugin/3.4.1");
            connection.setConnectTimeout(15000);
            connection.setReadTimeout(60000);
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                this.logger.debug("Failed to download: " + urlString + " - HTTP error code: " + responseCode);
                if (urlString.contains("JDA")) {
                    String fallbackUrl = "https://github.com/discord-jda/JDA/releases/download/v5.0.0-beta.18/JDA-5.0.0-beta.18-withDependencies.jar";
                    this.logger.debug("Trying fallback URL for JDA: " + fallbackUrl);
                    return this.downloadFile(fallbackUrl, outputFile);
                }
                if (responseCode == 404 && urlString.startsWith(MAVEN_CENTRAL)) {
                    String fileName = outputFile.getName();
                    String altFileName = fileName.replace(".jar", "-RELEASE.jar");
                    String altUrl = urlString.replace(fileName, altFileName);
                    this.logger.debug("Trying alternative URL format: " + altUrl);
                    return this.downloadFile(altUrl, new File(outputFile.getParentFile(), altFileName));
                }
                return false;
            }
            if (!FileUtils.ensureParentDirectoryExists(outputFile)) {
                this.logger.warning("Failed to create parent directory for download: " + outputFile.getAbsolutePath());
                return false;
            }
            Path tempFile = Files.createTempFile(outputFile.getName().replace(".jar", "-downloading-"), ".tmp", new FileAttribute[0]);
            long contentLength = connection.getContentLengthLong();
            this.logger.debug("Downloading " + outputFile.getName() + " (" + (String)(contentLength > 0L ? contentLength / 1024L + " KB" : "unknown size") + ")");
            try (InputStream in = connection.getInputStream();
                 ReadableByteChannel readChannel = Channels.newChannel(in);
                 FileOutputStream fileOS = new FileOutputStream(tempFile.toFile());
                 FileChannel writeChannel = fileOS.getChannel();){
                long totalBytes = contentLength;
                long bytesTransferred = 0L;
                int bufferSize = 0x100000;
                long position = 0L;
                long lastProgressUpdate = 0L;
                while (bytesTransferred < totalBytes || totalBytes == -1L) {
                    int progress;
                    long transferred = writeChannel.transferFrom(readChannel, position, bufferSize);
                    if (transferred == 0L) {
                        if (totalBytes == -1L) {
                            break;
                        }
                        try {
                            Thread.sleep(100L);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new IOException("Download interrupted");
                        }
                        if (++position % 30L != 0L) continue;
                        this.logger.debug("Download stalled, assuming completion");
                        break;
                    }
                    position += transferred;
                    long now = System.currentTimeMillis();
                    if (totalBytes <= 0x100000L || totalBytes == -1L || now - lastProgressUpdate <= 2000L || (progress = (int)((bytesTransferred += transferred) * 100L / totalBytes)) % 10 != 0 && progress != 99) continue;
                    this.logger.debug("Downloading " + outputFile.getName() + ": " + progress + "% complete");
                    lastProgressUpdate = now;
                }
            }
            if (Files.size(tempFile) == 0L) {
                this.logger.warning("Downloaded file is empty: " + outputFile.getName());
                Files.deleteIfExists(tempFile);
                return false;
            }
            Files.move(tempFile, outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            this.logger.debug("Successfully downloaded " + outputFile.getName() + " (" + outputFile.length() / 1024L + " KB)");
            return true;
        }
        catch (IOException e) {
            this.logger.warning("Error downloading " + urlString + ": " + e.getMessage());
            if (outputFile.exists() && outputFile.length() == 0L) {
                outputFile.delete();
            }
            return false;
        }
    }

    public String generateEnvironmentReport() {
        StringBuilder report = new StringBuilder();
        report.append("=== Discord Admin Plugin Environment Report ===\n");
        report.append("Java Version: ").append(System.getProperty("java.version")).append("\n");
        report.append("OS: ").append(System.getProperty("os.name")).append(" ").append(System.getProperty("os.version")).append("\n");
        report.append("Plugin Path: ").append(this.pluginPath).append("\n\n");
        report.append("== Rising World API Status ==\n");
        boolean apiAvailable = this.checkRisingWorldApiAvailable();
        report.append(apiAvailable ? "\u2713 Rising World API is available\n" : "\u2717 Rising World API is not available\n");
        report.append("\n== Libraries Status ==\n");
        File libFolder = new File(this.pluginPath + "/lib");
        if (!libFolder.exists()) {
            report.append("Lib folder does not exist at: ").append(libFolder.getAbsolutePath()).append("\n");
        } else {
            File[] libFiles = libFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".jar"));
            if (libFiles == null || libFiles.length == 0) {
                report.append("No libraries found in lib folder\n");
            } else {
                report.append("Found ").append(libFiles.length).append(" libraries:\n");
                for (Object object : libFiles) {
                    report.append("- ").append(((File)object).getName()).append(" (").append(((File)object).length() / 1024L).append(" KB)\n");
                }
            }
        }
        report.append("\n== Dependencies Check ==\n");
        for (Map.Entry entry : DEPENDENCIES.entrySet()) {
            String artifactId;
            String groupId;
            String version;
            String[] stringArray;
            String dependencyName = (String)entry.getKey();
            DependencyInfo info = (DependencyInfo)entry.getValue();
            if (info.getResolvedVersion() == null && (stringArray = info.getMavenCoordinates().split(":")).length == 2 && (version = this.resolveLatestVersion(groupId = stringArray[0], artifactId = stringArray[1], dependencyName)) != null) {
                info.setResolvedVersion(version);
            }
            try {
                Class.forName(info.getTestClass());
                report.append("\u2713 ").append(dependencyName).append(" (").append(info.getFullCoordinates()).append(")\n");
            }
            catch (ClassNotFoundException classNotFoundException) {
                report.append("\u2717 ").append(dependencyName).append(" (").append(info.getFullCoordinates()).append(")\n");
                String[] coordinates2 = info.getMavenCoordinates().split(":");
                if (coordinates2.length != 2) continue;
                String groupId2 = coordinates2[0];
                String artifactId2 = coordinates2[1];
                String version2 = info.getResolvedVersion();
                if (version2 != null) {
                    String mavenUrl = MAVEN_CENTRAL + groupId2.replace('.', '/') + "/" + artifactId2 + "/" + version2 + "/" + artifactId2 + "-" + version2 + ".jar";
                    report.append("  Download URL: ").append(mavenUrl).append("\n");
                    continue;
                }
                report.append("  Could not resolve latest version. Check manually at: ").append(MAVEN_CENTRAL).append(groupId2.replace('.', '/')).append("/").append(artifactId2).append("/\n");
            }
        }
        report.append("\n== Java Classpath ==\n");
        String classpath = System.getProperty("java.class.path");
        report.append(classpath).append("\n");
        report.append("\n== Network Connectivity Test ==\n");
        try {
            URL uRL = new URL(MAVEN_CENTRAL);
            HttpURLConnection conn = (HttpURLConnection)uRL.openConnection();
            conn.setConnectTimeout(5000);
            conn.setRequestMethod("HEAD");
            int responseCode = conn.getResponseCode();
            report.append("Maven Central connectivity: ").append((String)(responseCode == 200 ? "\u2713 Available" : "\u2717 Not available (Code: " + responseCode + ")")).append("\n");
        }
        catch (Exception exception) {
            report.append("Maven Central connectivity: \u2717 Error - ").append(exception.getMessage()).append("\n");
        }
        return report.toString();
    }

    static {
        DEPENDENCIES.put("JDA (Java Discord API)", new DependencyInfo("net.dv8tion:JDA", "net.dv8tion.jda.api.JDA", true));
        DEPENDENCIES.put("JSON-Simple", new DependencyInfo("com.googlecode.json-simple:json-simple", "org.json.simple.JSONObject", true));
        DEPENDENCIES.put("OkHttp", new DependencyInfo("com.squareup.okhttp3:okhttp", "okhttp3.Cache", true));
        DEPENDENCIES.put("SLF4J API", new DependencyInfo("org.slf4j:slf4j-api", "org.slf4j.Logger", true));
        DEPENDENCIES.put("SLF4J Simple", new DependencyInfo("org.slf4j:slf4j-simple", "org.slf4j.LoggerFactory", true));
        DEPENDENCIES.put("Jackson Core", new DependencyInfo("com.fasterxml.jackson.core:jackson-core", "com.fasterxml.jackson.core.JsonParser", true));
        DEPENDENCIES.put("Jackson ann", new DependencyInfo("com.fasterxml.jackson.core:jackson-annotations", "com.fasterxml.jackson.annotation.JsonView", true));
        DEPENDENCIES.put("Trove4J", new DependencyInfo("net.sf.trove4j:core", "gnu.trove.map.TLongObjectMap", true));
        DEPENDENCIES.put("kotlin-stdlib", new DependencyInfo("org.jetbrains.kotlin:kotlin-stdlib", "kotlin.jvm.internal.markers.KMappedMarker", true));
        DEPENDENCIES.put("tink", new DependencyInfo("com.google.crypto.tink:tink", "com.google.crypto.tink.CryptoFormat", true));
        DEPENDENCIES.put("Jackson Databine", new DependencyInfo("com.fasterxml.jackson.core:jackson-databind", "com.fasterxml.jackson.databind.AbstractTypeResolver", true));
        DEPENDENCIES.put("okio", new DependencyInfo("com.squareup.okio:okio", "okio.Source", true));
        DEPENDENCIES.put("okio jvm", new DependencyInfo("com.squareup.okio:okio-jvm", "okio.Buffer", true));
        DEPENDENCIES.put("neovisionaries", new DependencyInfo("com.neovisionaries:nv-websocket-client", "com.neovisionaries.ws.client.WebSocketFactory", true));
        DEPENDENCIES.put("collections4", new DependencyInfo("org.wso2.orbit.org.apache.commons:commons-collections4", "org.apache.commons.collections4.Bag", true));
    }

    private static class DependencyInfo {
        private final String mavenCoordinates;
        private final String testClass;
        private final boolean required;
        private String resolvedVersion;

        public DependencyInfo(String mavenCoordinates, String testClass, boolean required) {
            this.mavenCoordinates = mavenCoordinates;
            this.testClass = testClass;
            this.required = required;
            this.resolvedVersion = null;
        }

        public String getMavenCoordinates() {
            return this.mavenCoordinates;
        }

        public String getTestClass() {
            return this.testClass;
        }

        public boolean isRequired() {
            return this.required;
        }

        public String getResolvedVersion() {
            return this.resolvedVersion;
        }

        public void setResolvedVersion(String resolvedVersion) {
            this.resolvedVersion = resolvedVersion;
        }

        public String getFullCoordinates() {
            return this.resolvedVersion != null ? this.mavenCoordinates + ":" + this.resolvedVersion : this.mavenCoordinates;
        }
    }
}

