/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.BackupManager;
import com.example.risingworld.discordadmin.ConfigManager;
import com.example.risingworld.discordadmin.PluginLogger;
import com.example.risingworld.discordadmin.StatusManager;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import net.risingworld.api.Server;
import net.risingworld.api.objects.Player;

public class RestartManager {
    private final PluginLogger logger;
    private final StatusManager statusManager;
    private final ConfigManager configManager;
    private final Properties config;
    private ScheduledExecutorService scheduler;
    private ScheduledFuture<?> dailyRestartTask;
    private final AtomicBoolean locked = new AtomicBoolean(false);
    private final AtomicBoolean kicked = new AtomicBoolean(false);
    private String restartCommand;
    private int countdownMinutes = 30;
    private boolean restartEnabled = true;
    private TimeZone timeZone = TimeZone.getDefault();
    private static final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("HH:mm");
    private String restartMessagePrefix = "\u26a0\ufe0f **SERVER RESTART**";
    private int[] warningTimes = new int[]{20, 15, 10, 5, 3, 1};
    private BackupManager backupManager;
    private boolean preShutdownBackupEnabled;
    private String immediateAction;

    public RestartManager(PluginLogger logger, StatusManager statusManager, ConfigManager configManager, Properties config, String serverBasePath) {
        this.logger = logger;
        this.statusManager = statusManager;
        this.configManager = configManager;
        this.config = config;
        this.scheduler = Executors.newScheduledThreadPool(2);
        this.loadConfiguration();
        if (this.preShutdownBackupEnabled) {
            String backupPath = configManager.getConfigString(config, "backup.directory", null);
            this.backupManager = new BackupManager(logger, serverBasePath, backupPath);
            int maxBackups = configManager.getConfigInt(config, "backup.max_count", 10);
            this.backupManager.cleanupOldBackups(maxBackups);
        }
        if (this.restartEnabled) {
            String time = configManager.getConfigString(config, "restart.daily.time", "04:00");
            this.scheduleDailyRestart(time);
        } else {
            logger.debug("Daily restart is disabled in configuration");
        }
    }

    private void loadConfiguration() {
        if (this.config != null) {
            this.restartCommand = this.configManager.getConfigString(this.config, "restart.command", "");
            this.countdownMinutes = this.configManager.getConfigInt(this.config, "restart.countdown.minutes", 30);
            this.restartEnabled = this.configManager.getConfigBoolean(this.config, "restart.daily.enabled", true);
            this.preShutdownBackupEnabled = this.configManager.getConfigBoolean(this.config, "restart.pre_shutdown.backup", true);
            this.immediateAction = this.configManager.getConfigString(this.config, "restart.immediate_action", "lock");
            String timeZoneId = this.configManager.getConfigString(this.config, "restart.timezone", TimeZone.getDefault().getID());
            try {
                this.timeZone = TimeZone.getTimeZone(timeZoneId);
                this.logger.debug("Using timezone: " + this.timeZone.getID() + " (" + this.timeZone.getDisplayName() + ")");
            }
            catch (Exception e) {
                this.logger.warning("Invalid timezone ID: " + timeZoneId + ", using system default");
                this.timeZone = TimeZone.getDefault();
            }
            TIME_FORMAT.setTimeZone(this.timeZone);
            String warningTimesStr = this.configManager.getConfigString(this.config, "restart.warning.times", "20,15,10,5,3,1");
            try {
                List<String> warningStrings = Arrays.asList(warningTimesStr.split(","));
                if (!warningStrings.isEmpty()) {
                    int i;
                    this.warningTimes = new int[warningStrings.size()];
                    for (i = 0; i < warningStrings.size(); ++i) {
                        this.warningTimes[i] = Integer.parseInt(warningStrings.get(i).trim());
                    }
                    Arrays.sort(this.warningTimes);
                    for (i = 0; i < this.warningTimes.length / 2; ++i) {
                        int temp = this.warningTimes[i];
                        this.warningTimes[i] = this.warningTimes[this.warningTimes.length - 1 - i];
                        this.warningTimes[this.warningTimes.length - 1 - i] = temp;
                    }
                }
            }
            catch (Exception e) {
                this.logger.warning("Error parsing warning times, using defaults: " + e.getMessage());
                this.warningTimes = new int[]{20, 15, 10, 5, 3, 1};
            }
            this.restartMessagePrefix = this.configManager.getConfigString(this.config, "restart.broadcast.prefix", "\u26a0\ufe0f **SERVER RESTART**");
            if (this.countdownMinutes < 1) {
                this.countdownMinutes = 30;
                this.logger.warning("Invalid restart countdown minutes, using default of 30 minutes");
            }
        }
    }

    public void scheduleDailyRestart(String timeString) {
        if (!this.restartEnabled) {
            this.logger.debug("Daily restart is disabled, not scheduling");
            return;
        }
        this.locked.set(false);
        this.kicked.set(false);
        if (this.dailyRestartTask != null && !this.dailyRestartTask.isDone()) {
            this.dailyRestartTask.cancel(false);
        }
        try {
            Date restartTime = TIME_FORMAT.parse(timeString);
            Calendar restartCalendar = Calendar.getInstance(this.timeZone);
            Calendar now = Calendar.getInstance(this.timeZone);
            Calendar tempCal = Calendar.getInstance(this.timeZone);
            tempCal.setTime(restartTime);
            restartCalendar.set(11, tempCal.get(11));
            restartCalendar.set(12, tempCal.get(12));
            restartCalendar.set(13, 0);
            restartCalendar.set(14, 0);
            if (restartCalendar.before(now)) {
                restartCalendar.add(5, 1);
            }
            long delayMillis = restartCalendar.getTimeInMillis() - now.getTimeInMillis();
            long hours = delayMillis / 3600000L;
            long minutes = delayMillis / 60000L % 60L;
            this.logger.info("Daily server restart scheduled for: " + TIME_FORMAT.format(restartCalendar.getTime()) + " " + this.timeZone.getID());
            this.logger.debug("Time until restart: " + hours + " hours and " + minutes + " minutes");
            this.dailyRestartTask = this.scheduler.schedule(() -> {
                try {
                    if (!this.isRestartEnabled()) {
                        this.logger.info("Daily restart was canceled because restarts are now disabled");
                        this.scheduler.schedule(() -> this.scheduleDailyRestart(timeString), 1L, TimeUnit.MINUTES);
                        return;
                    }
                    Date executionTime = new Date();
                    this.logger.debug("Daily restart initiated at: " + TIME_FORMAT.format(executionTime));
                    this.locked.set(false);
                    this.kicked.set(false);
                    this.executeRestartSequence("Scheduled daily maintenance");
                    this.scheduler.schedule(() -> this.scheduleDailyRestart(timeString), 1L, TimeUnit.MINUTES);
                }
                catch (Exception e) {
                    this.logger.log(Level.SEVERE, "Error in daily restart task", e);
                    this.scheduler.schedule(() -> this.scheduleDailyRestart(timeString), 1L, TimeUnit.HOURS);
                }
            }, delayMillis, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Error scheduling daily restart", e);
            this.logger.info("Using default restart time (4:00 AM)");
            Calendar defaultRestart = Calendar.getInstance(this.timeZone);
            defaultRestart.add(5, 1);
            defaultRestart.set(11, 4);
            defaultRestart.set(12, 0);
            defaultRestart.set(13, 0);
            defaultRestart.set(14, 0);
            long defaultDelay = defaultRestart.getTimeInMillis() - System.currentTimeMillis();
            this.dailyRestartTask = this.scheduler.schedule(() -> {
                try {
                    if (this.isRestartEnabled()) {
                        this.executeRestartSequence("Scheduled daily maintenance");
                    }
                    this.scheduler.schedule(() -> this.scheduleDailyRestart("04:00"), 1L, TimeUnit.MINUTES);
                }
                catch (Exception ex) {
                    this.logger.log(Level.SEVERE, "Error in default restart task", ex);
                }
            }, defaultDelay, TimeUnit.MILLISECONDS);
        }
    }

    public boolean isRestartEnabled() {
        return this.restartEnabled;
    }

    public void setRestartEnabled(boolean enabled) {
        this.restartEnabled = enabled;
        if (this.config != null) {
            this.config.setProperty("restart.daily.enabled", String.valueOf(enabled));
            this.configManager.getMigrationManager().saveConfig(this.config, "config.properties", "Discord Admin Plugin Configuration - Updated Restart Settings");
        }
        if (enabled) {
            this.logger.info("Restarts have been enabled");
            if (this.dailyRestartTask == null || this.dailyRestartTask.isDone()) {
                String time = this.configManager.getConfigString(this.config, "restart.daily.time", "04:00");
                this.scheduleDailyRestart(time);
            }
        } else {
            this.logger.info("Restarts have been disabled");
        }
    }

    public boolean setRestartTime(String timeString) {
        try {
            TIME_FORMAT.parse(timeString);
        }
        catch (Exception e) {
            this.logger.warning("Invalid time format: " + timeString);
            return false;
        }
        if (this.config != null) {
            this.config.setProperty("restart.daily.time", timeString);
            this.configManager.getMigrationManager().saveConfig(this.config, "config.properties", "Discord Admin Plugin Configuration - Updated Restart Time");
        }
        if (this.restartEnabled) {
            this.scheduleDailyRestart(timeString);
        }
        this.logger.info("Set daily restart time to: " + timeString);
        return true;
    }

    public boolean setRestartTimezone(String timezoneId) {
        try {
            TimeZone newTimezone = TimeZone.getTimeZone(timezoneId);
            if (newTimezone.getID().equals("GMT") && !timezoneId.equals("GMT")) {
                this.logger.warning("Invalid timezone ID: " + timezoneId);
                return false;
            }
            this.timeZone = newTimezone;
            TIME_FORMAT.setTimeZone(this.timeZone);
            if (this.config != null) {
                this.config.setProperty("restart.timezone", this.timeZone.getID());
                this.configManager.getMigrationManager().saveConfig(this.config, "config.properties", "Discord Admin Plugin Configuration - Updated Timezone");
            }
            if (this.restartEnabled) {
                String time = this.configManager.getConfigString(this.config, "restart.daily.time", "04:00");
                this.scheduleDailyRestart(time);
            }
            this.logger.info("Set restart timezone to: " + this.timeZone.getID() + " (" + this.timeZone.getDisplayName() + ")");
            return true;
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error setting timezone", e);
            return false;
        }
    }

    public TimeZone getTimeZone() {
        return this.timeZone;
    }

    public String[] getAvailableTimezones() {
        return TimeZone.getAvailableIDs();
    }

    public String[] getCommonTimezones() {
        return new String[]{"GMT", "UTC", "America/Los_Angeles", "America/Denver", "America/Chicago", "America/New_York", "America/Anchorage", "Pacific/Honolulu", "Europe/London", "Europe/Paris", "Europe/Helsinki", "Asia/Tokyo", "Australia/Sydney", "Pacific/Auckland"};
    }

    public void scheduleRestart() {
        this.locked.set(false);
        this.kicked.set(false);
        this.executeRestartSequence("Administrator-initiated restart");
    }

    private void executeRestartSequence(String reason) {
        if (this.statusManager != null) {
            this.statusManager.setServerRestarting(this.countdownMinutes);
        }
        this.broadcastRestartWarning("Server will restart in " + this.countdownMinutes + " minutes for " + reason);
        this.logger.info("Server restart in " + this.countdownMinutes + " minutes. Reason: " + reason);
        for (int warningTime : this.warningTimes) {
            if (warningTime >= this.countdownMinutes) continue;
            long delay = (long)(this.countdownMinutes - warningTime) * 60L;
            this.scheduler.schedule(() -> {
                try {
                    if (warningTime > 1) {
                        this.broadcastRestartWarning("Server will restart in " + warningTime + " minutes.");
                        if (this.statusManager != null) {
                            this.statusManager.setServerRestarting(warningTime);
                        }
                        if (warningTime <= 2 && "lock".equals(this.immediateAction) && !this.locked.getAndSet(true)) {
                            Server.sendInputCommand("lock Restart in progress");
                            this.logger.debug("Server locked for restart");
                        } else if (warningTime <= 2 && "kick".equals(this.immediateAction) && !this.kicked.getAndSet(true)) {
                            Server.saveAll();
                            this.logger.debug("Saving all server data before restart");
                            for (Player player : Server.getAllPlayers()) {
                                player.kick("Server is restarting in " + warningTime + " minutes");
                            }
                            this.logger.debug("Kicked all players for restart");
                        }
                    } else {
                        this.broadcastRestartWarning("Server restarting in 1 minute. Please log out.");
                        if (this.statusManager != null) {
                            this.statusManager.setServerRestarting(warningTime);
                        }
                        if (!this.kicked.getAndSet(true)) {
                            Server.saveAll();
                            this.logger.debug("Saving all server data before restart");
                            for (Player player : Server.getAllPlayers()) {
                                player.kick("Server is restarting");
                            }
                            this.logger.debug("Kicked all players for restart");
                        }
                    }
                }
                catch (Exception e) {
                    this.logger.log(Level.SEVERE, "Error in restart warning", e);
                }
            }, delay, TimeUnit.SECONDS);
        }
        this.scheduler.schedule(() -> {
            try {
                this.broadcastRestartWarning("Server is now restarting. Goodbye!");
                if (this.statusManager != null) {
                    this.statusManager.setServerOffline();
                }
                Server.saveAll();
                if (this.preShutdownBackupEnabled && this.backupManager != null) {
                    try {
                        this.logger.info("Creating pre-shutdown backup...");
                        CompletableFuture<Boolean> backupFuture = this.backupManager.createBackup("Pre-restart");
                        boolean backupSuccess = backupFuture.get(300L, TimeUnit.SECONDS);
                        if (backupSuccess) {
                            this.logger.info("Pre-shutdown backup completed successfully");
                        } else {
                            this.logger.warning("Pre-shutdown backup did not complete successfully");
                        }
                    }
                    catch (Exception e) {
                        this.logger.log(Level.SEVERE, "Error during pre-shutdown backup", e);
                    }
                }
                this.executeRestartCommand();
                Server.shutdown();
            }
            catch (Exception e) {
                this.logger.log(Level.SEVERE, "Error in restart execution", e);
                Server.shutdown();
            }
        }, (long)this.countdownMinutes * 60L, TimeUnit.SECONDS);
    }

    public void restartNow(String reason) {
        this.locked.set(false);
        this.kicked.set(false);
        this.logger.info("Immediate server restart requested: " + reason);
        if (this.statusManager != null) {
            this.statusManager.setServerRestarting(0);
        }
        this.broadcastRestartWarning("Server is restarting NOW! " + reason);
        for (Player player : Server.getAllPlayers()) {
            player.kick("Server is restarting now: " + reason);
        }
        this.scheduler.schedule(() -> {
            try {
                Server.saveAll();
                if (this.preShutdownBackupEnabled && this.backupManager != null) {
                    try {
                        this.logger.info("Creating emergency pre-shutdown backup...");
                        CompletableFuture<Boolean> backupFuture = this.backupManager.createBackup("Emergency restart");
                        boolean backupSuccess = backupFuture.get(120L, TimeUnit.SECONDS);
                        if (backupSuccess) {
                            this.logger.info("Emergency backup completed successfully");
                        } else {
                            this.logger.warning("Emergency backup did not complete successfully");
                        }
                    }
                    catch (Exception e) {
                        this.logger.log(Level.SEVERE, "Error during emergency backup", e);
                    }
                }
                this.executeRestartCommand();
                Server.shutdown();
            }
            catch (Exception e) {
                this.logger.log(Level.SEVERE, "Error in emergency restart", e);
                Server.shutdown();
            }
        }, 5L, TimeUnit.SECONDS);
    }

    private void executeRestartCommand() {
        if (this.restartCommand == null || this.restartCommand.trim().isEmpty()) {
            this.logger.warning("No restart command specified");
            return;
        }
        try {
            this.logger.debug("Executing restart command: " + this.restartCommand);
            ProcessBuilder processBuilder = new ProcessBuilder(this.restartCommand.split(" "));
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            Thread outputThread = new Thread(() -> {
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        this.logger.debug("Restart process output: " + line);
                    }
                }
                catch (Exception e) {
                    this.logger.debug("Error reading restart process output: " + e.getMessage());
                }
            });
            outputThread.setName("RestartOutput-Reader");
            outputThread.setDaemon(true);
            outputThread.start();
            this.logger.info("Restart command initiated");
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Error executing restart command", e);
        }
    }

    private void broadcastRestartWarning(String message) {
        Server.broadcastTextMessage(this.restartMessagePrefix + " " + message);
        if (this.statusManager != null) {
            this.statusManager.updateStatusMessage(true, message, true);
        }
    }

    public void shutdown() {
        try {
            if (this.dailyRestartTask != null && !this.dailyRestartTask.isDone()) {
                this.dailyRestartTask.cancel(false);
            }
            if (this.scheduler != null) {
                this.scheduler.shutdown();
                if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                }
                this.logger.debug("Restart manager scheduler shut down");
            }
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Error shutting down restart manager", e);
        }
    }
}

