/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.PluginLogger;
import com.example.risingworld.discordadmin.StatusManager;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.JDABuilder;
import net.dv8tion.jda.api.Permission;
import net.dv8tion.jda.api.entities.Activity;
import net.dv8tion.jda.api.entities.Message;
import net.dv8tion.jda.api.entities.channel.concrete.TextChannel;
import net.dv8tion.jda.api.entities.channel.middleman.GuildChannel;
import net.dv8tion.jda.api.events.session.ReadyEvent;
import net.dv8tion.jda.api.events.session.SessionDisconnectEvent;
import net.dv8tion.jda.api.events.session.SessionRecreateEvent;
import net.dv8tion.jda.api.hooks.ListenerAdapter;
import net.dv8tion.jda.api.requests.GatewayIntent;
import net.risingworld.api.objects.Player;

public class DiscordService {
    private final PluginLogger logger;
    private final String pluginPath;
    private final Properties config;
    private JDA jda;
    private TextChannel adminChannel;
    private String token;
    private String adminChannelId;
    private boolean connected = false;
    private ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
    private StatusManager statusManager;

    public DiscordService(String pluginPath, PluginLogger logger, Properties config) {
        this.pluginPath = pluginPath;
        this.logger = logger;
        this.config = config;
        this.token = this.getConfigValue("discord.token", "");
        this.adminChannelId = this.getConfigValue("discord.admin.channel", "");
    }

    public boolean initialize() {
        if (this.token == null || this.token.trim().isEmpty()) {
            this.logger.warning("Discord token not set. Please configure 'discord.token' in config.properties.");
            return false;
        }
        try {
            this.logger.info("Connecting to Discord...");
            JDABuilder builder = JDABuilder.createDefault((String)this.token).enableIntents(GatewayIntent.MESSAGE_CONTENT, new GatewayIntent[]{GatewayIntent.GUILD_MESSAGES, GatewayIntent.GUILD_MEMBERS, GatewayIntent.GUILD_INVITES}).setAutoReconnect(true);
            String activityText = this.getConfigValue("discord.bot.activity", "Watching Rising World");
            if (activityText != null && !activityText.isEmpty()) {
                builder.setActivity(Activity.watching((String)activityText));
            }
            builder.addEventListeners(new Object[]{new ListenerAdapter(){

                public void onSessionDisconnect(SessionDisconnectEvent event) {
                    if (!event.isClosedByServer()) {
                        DiscordService.this.logger.warning("Discord connection lost. Auto-reconnect is enabled and will attempt to restore connection.");
                    }
                }

                public void onSessionRecreate(SessionRecreateEvent event) {
                    DiscordService.this.logger.info("Discord connection has been restored!");
                    if (DiscordService.this.statusManager != null) {
                        DiscordService.this.scheduler.schedule(() -> DiscordService.this.statusManager.initialize(), 5L, TimeUnit.SECONDS);
                    }
                }
            }});
            Thread.setDefaultUncaughtExceptionHandler((thread, exception) -> {
                if (thread.getName().startsWith("JDA")) {
                    this.logger.log(Level.SEVERE, "Uncaught exception in JDA thread: " + thread.getName(), exception);
                    boolean notifyErrors = Boolean.parseBoolean(this.getConfigValue("discord.error.notification.enabled", "true"));
                    if (notifyErrors && this.adminChannel != null) {
                        this.adminChannel.sendMessage((CharSequence)("\u26a0\ufe0f **JDA ERROR**: An error occurred in the Discord connection: " + exception.getMessage() + "\nCheck server logs for details.")).queue();
                    }
                }
            });
            this.jda = builder.build();
            boolean connectionSuccess = this.waitForConnection(30);
            if (!connectionSuccess) {
                this.logger.severe("Failed to connect to Discord - timeout occurred.");
                if (this.token.length() < 50) {
                    this.logger.severe("The Discord token appears to be invalid (too short). Please check your configuration.");
                } else {
                    this.logger.severe("Connection timed out. Possible causes: network issues, invalid token, or Discord service disruption.");
                }
                return false;
            }
            this.logger.info("Connected to Discord successfully");
            this.connected = true;
            if (this.adminChannelId != null && !this.adminChannelId.trim().isEmpty()) {
                this.adminChannel = this.jda.getTextChannelById(this.adminChannelId);
                if (this.adminChannel == null) {
                    this.logger.warning("Could not find admin channel with ID: " + this.adminChannelId);
                    this.logger.warning("Possible causes: channel deleted, bot lacks access permissions, or ID is incorrect.");
                } else {
                    boolean canSendMessages = this.adminChannel.canTalk();
                    boolean canPinMessages = this.adminChannel.getGuild().getSelfMember().hasPermission((GuildChannel)this.adminChannel, new Permission[]{Permission.MESSAGE_MANAGE});
                    if (!canSendMessages) {
                        this.logger.warning("Bot does not have permission to send messages in the admin channel.");
                    }
                    if (!canPinMessages) {
                        this.logger.warning("Bot does not have permission to pin messages in the admin channel.");
                    }
                    this.logger.debug("Successfully connected to Discord admin channel: " + this.adminChannel.getName());
                }
            } else {
                this.logger.warning("Admin channel ID not set. Please set 'discord.admin.channel' in config.properties.");
            }
            return true;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Invalid Discord token or connection error. Please check your configuration.", e);
            this.shutdown();
            return false;
        }
    }

    private boolean waitForConnection(int timeoutSeconds) {
        try {
            final CompletableFuture future = new CompletableFuture();
            this.jda.addEventListener(new Object[]{new ListenerAdapter(this){

                public void onReady(ReadyEvent event) {
                    future.complete(null);
                }
            }});
            future.get(timeoutSeconds, TimeUnit.SECONDS);
            return true;
        }
        catch (InterruptedException e) {
            this.logger.log(Level.SEVERE, "Interrupted while connecting to Discord", e);
            Thread.currentThread().interrupt();
            return false;
        }
        catch (TimeoutException e) {
            this.logger.severe("Timeout while waiting for Discord connection");
            return false;
        }
        catch (ExecutionException e) {
            this.logger.log(Level.SEVERE, "Error while connecting to Discord", e);
            return false;
        }
    }

    public void addEventListener(ListenerAdapter listener) {
        if (this.jda != null) {
            this.jda.addEventListener(new Object[]{listener});
        } else {
            this.logger.warning("Cannot add event listener: JDA is not initialized");
        }
    }

    public CompletableFuture<Message> sendAdminMessage(String message) {
        CompletableFuture<Message> result = new CompletableFuture<Message>();
        if (this.adminChannel == null) {
            result.completeExceptionally(new IllegalStateException("Admin channel not configured"));
            return result;
        }
        try {
            this.adminChannel.sendMessage((CharSequence)message).queue(message_ -> result.complete((Message)message_), error -> {
                this.logger.warning("Failed to send message to admin channel: " + error.getMessage());
                result.completeExceptionally((Throwable)error);
            });
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error sending admin message", e);
            result.completeExceptionally(e);
        }
        return result;
    }

    public void sendChatMessage(String formattedMessage) {
        if (this.adminChannel != null) {
            this.adminChannel.sendMessage((CharSequence)formattedMessage).queue();
        }
    }

    public void sendCommandLog(String playerName, String command) {
        if (this.adminChannel != null) {
            this.adminChannel.sendMessage((CharSequence)("Command executed in-game by " + playerName + ": " + command)).queue();
        }
    }

    public CompletableFuture<Message> sendPlayerJoinMessage(String playerName) {
        if (this.adminChannel == null) {
            return CompletableFuture.failedFuture(new IllegalStateException("Admin channel not configured"));
        }
        CompletableFuture<Message> result = new CompletableFuture<Message>();
        String joinFormat = this.getConfigValue("message.join.format", "\ud83d\udfe2 **%player% joined the server**");
        String message = joinFormat.replace("%player%", playerName);
        this.adminChannel.sendMessage((CharSequence)message).queue(msg -> result.complete((Message)msg), error -> result.completeExceptionally((Throwable)error));
        return result;
    }

    public CompletableFuture<Message> sendPlayerLeaveMessage(String playerName) {
        if (this.adminChannel == null) {
            return CompletableFuture.failedFuture(new IllegalStateException("Admin channel not configured"));
        }
        CompletableFuture<Message> result = new CompletableFuture<Message>();
        String leaveFormat = this.getConfigValue("message.leave.format", "\ud83d\udd34 **%player% left the server**");
        String message = leaveFormat.replace("%player%", playerName);
        this.adminChannel.sendMessage((CharSequence)message).queue(msg -> result.complete((Message)msg), error -> result.completeExceptionally((Throwable)error));
        return result;
    }

    public boolean sendDiscordInvite(Player player, String inviteUrl) {
        if (player == null || inviteUrl == null || ((String)inviteUrl).isEmpty()) {
            return false;
        }
        try {
            inviteUrl = "https://discord.gg/" + (String)inviteUrl;
            player.connectToDiscord((String)inviteUrl);
            this.logger.debug("Sent Discord invite " + (String)inviteUrl + " to player: " + player.getName());
            return true;
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error sending Discord invite to player: " + player.getName(), e);
            return false;
        }
    }

    public JDA getJDA() {
        return this.jda;
    }

    public TextChannel getAdminChannel() {
        return this.adminChannel;
    }

    public boolean isConnected() {
        return this.connected && this.jda != null && this.jda.getStatus() == JDA.Status.CONNECTED;
    }

    public void shutdown() {
        if (this.jda != null) {
            try {
                this.logger.debug("Waiting for pending Discord operations to complete...");
                Thread.sleep(1000L);
                this.logger.debug("Shutting down Discord connection...");
                this.jda.shutdown();
                if (this.jda.awaitShutdown(15L, TimeUnit.SECONDS)) {
                    this.logger.debug("Discord connection shut down successfully.");
                } else {
                    this.logger.warning("Discord connection shutdown timed out.");
                    this.jda.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.logger.log(Level.SEVERE, "Interrupted while shutting down Discord connection", e);
                Thread.currentThread().interrupt();
                if (this.jda != null) {
                    this.jda.shutdownNow();
                }
            }
            catch (Exception e) {
                this.logger.log(Level.SEVERE, "Error shutting down Discord connection", e);
                if (this.jda != null) {
                    try {
                        this.jda.shutdownNow();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
            }
            finally {
                this.connected = false;
                this.jda = null;
                this.logger.info("Discord connection terminated.");
            }
        }
    }

    public void setStatusManager(StatusManager statusManager) {
        this.statusManager = statusManager;
    }

    private String getConfigValue(String key, String defaultValue) {
        if (this.config == null) {
            return defaultValue;
        }
        return this.config.getProperty(key, defaultValue);
    }
}

