/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.FileUtils;
import com.example.risingworld.discordadmin.PluginLogger;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class PlayerConnectionLogger {
    private static final String LOG_FILE_PREFIX = "connections_";
    private static final String LOG_FILE_SUFFIX = ".log";
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
    private static final SimpleDateFormat TIMESTAMP_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private final File logDirectory;
    private final PluginLogger logger;
    private File currentLogFile;
    private final int maxLogFiles;
    private final ConcurrentHashMap<String, Long> connectionTimes = new ConcurrentHashMap();

    public PlayerConnectionLogger(String logDirectoryPath, PluginLogger logger) {
        this.logger = logger;
        this.maxLogFiles = 30;
        this.logDirectory = new File(logDirectoryPath);
        if (FileUtils.ensureDirectoryExists(this.logDirectory)) {
            logger.info("Player connection log directory available at: " + this.logDirectory.getAbsolutePath());
        } else {
            logger.warning("Failed to create player connection log directory: " + this.logDirectory.getAbsolutePath());
        }
        this.currentLogFile = this.getCurrentLogFile();
        this.cleanupOldLogFiles();
    }

    public PlayerConnectionLogger(String pluginPath, PluginLogger logger, Properties config) {
        this.logger = logger;
        String logDirName = config.getProperty("player.connection.log.directory", "player_connections");
        this.maxLogFiles = Integer.parseInt(config.getProperty("logs.max_files", "30"));
        this.logDirectory = new File(pluginPath, logDirName);
        if (FileUtils.ensureDirectoryExists(this.logDirectory)) {
            logger.info("Player connection log directory available at: " + this.logDirectory.getAbsolutePath());
        } else {
            logger.warning("Failed to create player connection log directory: " + this.logDirectory.getAbsolutePath());
        }
        this.currentLogFile = this.getCurrentLogFile();
        this.cleanupOldLogFiles();
    }

    public synchronized void logPlayerConnect(String playerName, String uid) {
        long connectionTime = System.currentTimeMillis();
        this.connectionTimes.put(playerName.toLowerCase(), connectionTime);
        try {
            this.checkAndUpdateLogFile();
            if (!FileUtils.ensureParentDirectoryExists(this.currentLogFile)) {
                this.logger.warning("Failed to create parent directory for log file: " + this.currentLogFile.getAbsolutePath());
                return;
            }
            try (PrintWriter out = new PrintWriter(new FileWriter(this.currentLogFile, true));){
                String timestamp = TIMESTAMP_FORMAT.format(new Date(connectionTime));
                out.println(String.format("[%s] CONNECT: %s (UID: %s)", timestamp, playerName, uid));
            }
        }
        catch (IOException e) {
            this.logger.log(Level.SEVERE, "Failed to log player connection", e);
        }
    }

    public synchronized void logPlayerDisconnect(String playerName, String uid) {
        long disconnectTime = System.currentTimeMillis();
        String playerNameLower = playerName.toLowerCase();
        Long connectionTime = this.connectionTimes.remove(playerNameLower);
        try {
            this.checkAndUpdateLogFile();
            if (!FileUtils.ensureParentDirectoryExists(this.currentLogFile)) {
                this.logger.warning("Failed to create parent directory for log file: " + this.currentLogFile.getAbsolutePath());
                return;
            }
            try (PrintWriter out = new PrintWriter(new FileWriter(this.currentLogFile, true));){
                String timestamp = TIMESTAMP_FORMAT.format(new Date(disconnectTime));
                if (connectionTime != null) {
                    long sessionDuration = disconnectTime - connectionTime;
                    out.println(String.format("[%s] DISCONNECT: %s (UID: %s) - Session Duration: %s", timestamp, playerName, uid, this.formatDuration(sessionDuration)));
                } else {
                    out.println(String.format("[%s] DISCONNECT: %s (UID: %s) - Session duration unknown", timestamp, playerName, uid));
                }
            }
        }
        catch (IOException e) {
            this.logger.log(Level.SEVERE, "Failed to log player disconnection", e);
        }
    }

    private String formatDuration(long milliseconds) {
        long seconds = milliseconds / 1000L;
        long minutes = seconds / 60L;
        long hours = minutes / 60L;
        long days = hours / 24L;
        hours %= 24L;
        minutes %= 60L;
        seconds %= 60L;
        StringBuilder sb = new StringBuilder();
        if (days > 0L) {
            sb.append(days).append("d ");
        }
        if (hours > 0L || days > 0L) {
            sb.append(hours).append("h ");
        }
        if (minutes > 0L || hours > 0L || days > 0L) {
            sb.append(minutes).append("m ");
        }
        sb.append(seconds).append("s");
        return sb.toString();
    }

    private synchronized void checkAndUpdateLogFile() {
        String today = DATE_FORMAT.format(new Date());
        String currentFileName = LOG_FILE_PREFIX + today + LOG_FILE_SUFFIX;
        if (this.currentLogFile == null || !this.currentLogFile.getName().equals(currentFileName)) {
            this.currentLogFile = new File(this.logDirectory, currentFileName);
            this.logger.fine("Updated current log file to: " + this.currentLogFile.getName());
        }
    }

    private File getCurrentLogFile() {
        String dateStamp = DATE_FORMAT.format(new Date());
        return new File(this.logDirectory, LOG_FILE_PREFIX + dateStamp + LOG_FILE_SUFFIX);
    }

    private void cleanupOldLogFiles() {
        try {
            if (!this.logDirectory.exists() || !this.logDirectory.isDirectory()) {
                return;
            }
            File[] logFiles = this.logDirectory.listFiles(new FileFilter(this){

                @Override
                public boolean accept(File file) {
                    String name = file.getName();
                    return name.startsWith(PlayerConnectionLogger.LOG_FILE_PREFIX) && name.endsWith(PlayerConnectionLogger.LOG_FILE_SUFFIX);
                }
            });
            if (logFiles == null || logFiles.length <= this.maxLogFiles) {
                return;
            }
            ArrayList sortedFiles = new ArrayList();
            Collections.addAll(sortedFiles, logFiles);
            Collections.sort(sortedFiles, Comparator.comparingLong(File::lastModified));
            int filesToDelete = sortedFiles.size() - this.maxLogFiles;
            for (int i = 0; i < filesToDelete; ++i) {
                File oldFile = (File)sortedFiles.get(i);
                if (oldFile.delete()) {
                    this.logger.info("Deleted old log file: " + oldFile.getName());
                    continue;
                }
                this.logger.warning("Failed to delete old log file: " + oldFile.getName());
            }
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error cleaning up old log files", e);
        }
    }

    public List<String> getLogsForDate(String date, int maxLines) {
        ArrayList<String> result = new ArrayList<String>();
        if (!date.matches("\\d{4}-\\d{2}-\\d{2}")) {
            this.logger.warning("Invalid date format: " + date);
            result.add("Invalid date format. Use YYYY-MM-DD");
            return result;
        }
        File logFile = new File(this.logDirectory, LOG_FILE_PREFIX + date + LOG_FILE_SUFFIX);
        if (!logFile.exists()) {
            result.add("No logs found for date: " + date);
            return result;
        }
        try {
            int startLine;
            List<String> allLines = Files.readAllLines(logFile.toPath());
            for (int i = startLine = Math.max(0, allLines.size() - maxLines); i < allLines.size(); ++i) {
                result.add(allLines.get(i));
            }
        }
        catch (IOException e) {
            this.logger.log(Level.SEVERE, "Error reading log file for date: " + date, e);
            result.add("Error reading log file: " + e.getMessage());
        }
        return result;
    }

    public List<String> searchPlayerLogs(String playerName, int maxResults) {
        ArrayList<String> results = new ArrayList<String>();
        String searchTerm = playerName.toLowerCase();
        try {
            File[] logFiles = this.logDirectory.listFiles(file -> {
                String name = file.getName();
                return name.startsWith(LOG_FILE_PREFIX) && name.endsWith(LOG_FILE_SUFFIX);
            });
            if (logFiles == null || logFiles.length == 0) {
                results.add("No log files found");
                return results;
            }
            ArrayList<File> sortedFiles = new ArrayList<File>();
            Collections.addAll(sortedFiles, logFiles);
            sortedFiles.sort(Comparator.comparingLong(File::lastModified).reversed());
            int foundResults = 0;
            Pattern pattern = Pattern.compile("\\[.*\\] (?:CONNECT|DISCONNECT): (.*?) \\(UID:", 2);
            block2: for (File file2 : sortedFiles) {
                if (foundResults >= maxResults) break;
                List<String> lines = Files.readAllLines(file2.toPath());
                String fileDate = file2.getName().replace(LOG_FILE_PREFIX, "").replace(LOG_FILE_SUFFIX, "");
                for (String line : lines) {
                    String playerInLog;
                    Matcher matcher = pattern.matcher(line);
                    if (!matcher.find() || !(playerInLog = matcher.group(1).toLowerCase()).contains(searchTerm)) continue;
                    results.add("[" + fileDate + "] " + line);
                    if (++foundResults < maxResults) continue;
                    continue block2;
                }
            }
            if (results.isEmpty()) {
                results.add("No log entries found for player: " + playerName);
            }
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Error searching player logs", e);
            results.add("Error searching logs: " + e.getMessage());
        }
        return results;
    }
}

