/*
 * Decompiled with CFR 0.152.
 */
package com.example.risingworld.discordadmin;

import com.example.risingworld.discordadmin.FileUtils;
import com.example.risingworld.discordadmin.PluginLogger;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.concurrent.CompletableFuture;
import java.util.logging.Level;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.risingworld.api.Server;
import net.risingworld.api.events.EventMethod;

public class BackupManager {
    private final PluginLogger logger;
    private final String serverBasePath;
    private final String backupPath;
    private final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
    private volatile boolean saveInProgress = false;

    public BackupManager(PluginLogger logger, String serverBasePath, String configBackupPath) {
        this.logger = logger;
        this.serverBasePath = serverBasePath;
        this.backupPath = configBackupPath != null && !configBackupPath.isEmpty() ? configBackupPath : serverBasePath + File.separator + "backups";
        File backupDir = new File(this.backupPath);
        if (FileUtils.ensureDirectoryExists(backupDir)) {
            logger.info("Backup directory available at " + backupDir.getAbsolutePath());
        } else {
            logger.warning("Failed to create backup directory at: " + backupDir.getAbsolutePath());
        }
    }

    public CompletableFuture<Boolean> createBackup(String reason) {
        CompletableFuture<Boolean> result = new CompletableFuture<Boolean>();
        String timestamp = this.dateFormat.format(new Date());
        String backupFileName = "backup_" + timestamp + ".zip";
        File backupFile = new File(this.backupPath, backupFileName);
        if (!FileUtils.ensureParentDirectoryExists(backupFile)) {
            this.logger.severe("Unable to create parent directory for backup file: " + backupFile.getAbsolutePath());
            result.completeExceptionally(new IOException("Failed to create backup directory"));
            return result;
        }
        this.logger.info("Starting server backup: " + reason);
        this.saveInProgress = true;
        Server.saveAll();
        Thread backupThread = new Thread(() -> {
            try {
                int maxWaitSeconds = 30;
                for (int waited = 0; this.saveInProgress && waited < maxWaitSeconds; ++waited) {
                    Thread.sleep(1000L);
                }
                this.logger.info("Save completed, creating backup archive: " + backupFileName);
                File worldsDir = new File(this.serverBasePath + File.separator + "Worlds");
                File pluginsDir = new File(this.serverBasePath + File.separator + "Plugins");
                File configDir = new File(this.serverBasePath + File.separator + "Permissions");
                File serverPropertiesFile = new File(this.serverBasePath + File.separator + "server.properties");
                try (ZipOutputStream zipOut = new ZipOutputStream(Files.newOutputStream(backupFile.toPath(), new OpenOption[0]));){
                    if (worldsDir.exists()) {
                        this.addDirToZip(zipOut, worldsDir, "Worlds");
                    }
                    if (pluginsDir.exists()) {
                        this.addDirToZip(zipOut, pluginsDir, "Plugins");
                    }
                    if (configDir.exists()) {
                        this.addDirToZip(zipOut, configDir, "Config");
                    }
                    if (serverPropertiesFile.exists()) {
                        ZipEntry fileEntry = new ZipEntry("server.properties");
                        zipOut.putNextEntry(fileEntry);
                        Files.copy(serverPropertiesFile.toPath(), zipOut);
                        zipOut.closeEntry();
                        this.logger.info("Added server.properties to backup");
                    } else {
                        this.logger.warning("server.properties file not found at " + serverPropertiesFile.getAbsolutePath());
                    }
                    this.logger.info("Backup completed successfully: " + backupFile.getAbsolutePath());
                    result.complete(true);
                }
            }
            catch (Exception e) {
                this.logger.log(Level.SEVERE, "Error creating backup", e);
                result.completeExceptionally(e);
            }
        }, "Backup-Thread");
        backupThread.setDaemon(true);
        backupThread.start();
        return result;
    }

    private void addDirToZip(ZipOutputStream zipOut, File dirToZip, String entryPrefix) throws IOException {
        if (dirToZip == null || !dirToZip.exists() || !dirToZip.isDirectory()) {
            this.logger.warning("Cannot add directory to zip: " + (dirToZip == null ? "null" : dirToZip.getAbsolutePath()));
            return;
        }
        File[] files = dirToZip.listFiles();
        if (files == null) {
            this.logger.warning("Cannot list files in directory: " + dirToZip.getAbsolutePath());
            return;
        }
        for (File file : files) {
            String entryPath = entryPrefix + "/" + file.getName();
            if (file.isDirectory()) {
                ZipEntry dirEntry = new ZipEntry(entryPath + "/");
                zipOut.putNextEntry(dirEntry);
                zipOut.closeEntry();
                this.addDirToZip(zipOut, file, entryPath);
                continue;
            }
            try {
                ZipEntry fileEntry = new ZipEntry(entryPath);
                zipOut.putNextEntry(fileEntry);
                Files.copy(file.toPath(), zipOut);
                zipOut.closeEntry();
            }
            catch (IOException e) {
                this.logger.warning("Error adding file to backup: " + file.getAbsolutePath() + " - " + e.getMessage());
            }
        }
    }

    @EventMethod
    public void onServerSave() {
        this.saveInProgress = false;
    }

    public void cleanupOldBackups(int maxBackups) {
        if (maxBackups <= 0) {
            return;
        }
        try {
            File backupDir = new File(this.backupPath);
            if (!backupDir.exists() || !backupDir.isDirectory()) {
                this.logger.warning("Backup directory does not exist at: " + this.backupPath);
                return;
            }
            File[] backupFiles = backupDir.listFiles((dir, name) -> name.startsWith("backup_") && name.endsWith(".zip"));
            if (backupFiles == null || backupFiles.length <= maxBackups) {
                return;
            }
            Arrays.sort(backupFiles, (f1, f2) -> Long.compare(f1.lastModified(), f2.lastModified()));
            int filesToDelete = backupFiles.length - maxBackups;
            for (int i = 0; i < filesToDelete; ++i) {
                File oldBackup = backupFiles[i];
                if (oldBackup.delete()) {
                    this.logger.info("Deleted old backup: " + oldBackup.getName());
                    continue;
                }
                this.logger.warning("Failed to delete old backup: " + oldBackup.getName());
            }
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error cleaning up old backups", e);
        }
    }

    public double getBackupsTotalSizeMB() {
        File backupDir = new File(this.backupPath);
        if (!backupDir.exists() || !backupDir.isDirectory()) {
            return 0.0;
        }
        File[] backupFiles = backupDir.listFiles((dir, name) -> name.startsWith("backup_") && name.endsWith(".zip"));
        if (backupFiles == null) {
            return 0.0;
        }
        long totalSizeBytes = 0L;
        for (File file : backupFiles) {
            totalSizeBytes += file.length();
        }
        return (double)totalSizeBytes / 1048576.0;
    }
}

