package net.risingworld.api.example.staticmodels;

import java.io.File;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import net.risingworld.api.Plugin;
import net.risingworld.api.Timer;
import net.risingworld.api.database.Database;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.player.PlayerChangeEquippedItemEvent;
import net.risingworld.api.events.player.PlayerConnectEvent;
import net.risingworld.api.events.player.PlayerCraftItemEvent;
import net.risingworld.api.events.player.PlayerElementHitEvent;
import net.risingworld.api.events.player.PlayerKeyEvent;
import net.risingworld.api.events.player.gui.PlayerSelectFileEvent;
import net.risingworld.api.gui.Font;
import net.risingworld.api.gui.GuiFileBrowser;
import net.risingworld.api.gui.GuiLabel;
import net.risingworld.api.gui.GuiPanel;
import net.risingworld.api.gui.PivotPosition;
import net.risingworld.api.objects.Item;
import net.risingworld.api.objects.Player;
import net.risingworld.api.objects.custom.CustomItem;
import net.risingworld.api.objects.custom.CustomRecipe;
import net.risingworld.api.utils.Animation;
import net.risingworld.api.utils.BoundingInformation;
import net.risingworld.api.utils.CollisionShape;
import net.risingworld.api.utils.CollisionType;
import net.risingworld.api.utils.Definitions;
import net.risingworld.api.utils.ImageInformation;
import net.risingworld.api.utils.KeyInput;
import net.risingworld.api.utils.ModelInformation;
import net.risingworld.api.utils.Quaternion;
import net.risingworld.api.utils.Utils;
import net.risingworld.api.utils.Vector3f;
import net.risingworld.api.worldelements.World3DModel;
import net.risingworld.api.worldelements.WorldElement;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * 
 * @author red51
 */

public class StaticModelLoader extends Plugin implements Listener{
    
    //We store all created elements in an SQLite database, so we keep a reference
    //to our database connector
    private Database database;
    
    //This list holds all world elements we create
    private ArrayList<World3DModel> world3DModels = new ArrayList<>();
    
    private final String ATTRIBUTE_TIMER = "example.staticmodelloader.timer";
    private final String ATTRIBUTE_MODELPREVIEW = "example.staticmodelloader.modelpreview";
    private final String ATTRIBUTE_STRENGTH = "example.staticmodelloader.strength";
    
    private final String ATTRIBUTE_FILEBROWSER = "example.staticmodelloader.player.filebrowser";
    private final String ATTRIBUTE_FILEBROWSER_BACKGROUND = "example.staticmodelloader.player.filebrowser.background";
    private final String ATTRIBUTE_FILEBROWSER_HEADLINE = "example.staticmodelloader.player.filebrowser.header";
    
    private final String ATTRIBUTE_PENDING_MODEL = "example.staticmodelloader.player.upload.pendingmodel";
    private final String ATTRIBUTE_PENDING_TEXTURE = "example.staticmodelloader.player.upload.pendingtexture";
    
    //Final variable (constant) to define the UUID of the item (has to be unique!)
    public final String CUSTOMITEM_UUID = "net.risingworld.api.example.staticmodels.3dmodelblueprint";
    
    private final int DEFAULT_ELEMENT_STRENGTH = 500;
    
    //This bitmask is used for the player raycasts (see below). It determines what
    //types of collisions the raycast takes into account. In this case, we want the
    //ray to collide with the terrain, constructions (e.g. blocks and planks), objects
    //(e.g. furniture) and trees
    private final int COLLISION_BITMASK = CollisionType.getBitmask(CollisionType.TERRAIN, CollisionType.CONSTRUCTIONS, CollisionType.OBJECTS, CollisionType.TREES);
    
    @Override
    public void onEnable(){
        //Get database connection (creates a new "WorldModels.db" database in the world folder)
        database = getSQLiteConnection(getWorld().getWorldFolder() + "/" + "WorldModels.db");
        
        //We create a new table (only if it does not exist) in our database to store all 
        //relevant model information there
        database.execute("CREATE TABLE IF NOT EXISTS `models` (`ID` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, `model` VARCHAR(255), `texture` VARCHAR(255), `playeruid` BIGINT, `position` VARCHAR(255), `rotation` VARCHAR(255), `size` REAL, `collision` TINYINT)");
        
        //Basically this is only relevant if there was already a "models" table:
        //Get all entries from that table. We use a try-with-resources block, i.e. a
        //try-catch-block where the ResultSet is declared in the head - so if an exception
        //occurs, the ResultSet gets closed automatically
        try(ResultSet result = database.executeQuery("SELECT * FROM `models`")){
            //Iterate through all rows
            while(result.next()){
                //We create a separate try-catch-block for every result - so if there is a broken
                //entry in our table, the game will still load the other models
                try{
                    //Create the model- and texture information from information we get from the database
                    String path = getPath() + "/Models/" + result.getLong("playeruid") + "/";
                    File modelFile = new File(path + result.getString("model"));
                    File imageFile = new File(path + result.getString("texture"));
                    
                    if(modelFile.exists() && imageFile.exists()){
                        ModelInformation model = new ModelInformation(path + result.getString("model"));
                        ImageInformation texture = new ImageInformation(path + result.getString("texture"));

                        //Read the position, rotation and size that was stored in the database
                        Vector3f position = new Vector3f().fromString(result.getString("position"));
                        Quaternion rotation = new Quaternion().fromString(result.getString("rotation"));

                        //Create a new World3DModel instance
                        World3DModel world3DModel = createNewWorld3DModel(model, texture, result.getFloat("size"), position, rotation, result.getBoolean("collision"));
                        
                        //Add the element to our world3DModels list (which contains all world elements we've created)
                        world3DModels.add(world3DModel);
                    }
                }
                catch(Exception e){
                    //...
                }
            }
        }
        //If an exception occurs (or more precisely, we only check if an SQLException occurs), we catch it here
        catch(SQLException e){
            //Just print the whole stack trace, it's easier for debugging...
            e.printStackTrace();
        }
        
        //Create custom item (used to place 3d models)
        CustomItem item = new CustomItem(CUSTOMITEM_UUID, "modelblueprint");
        
        //Load the model, texture and icon (load from resource, i.e. from the plugin jar)
        ModelInformation model = new ModelInformation(this, "/resources/item.j3o");
        ImageInformation texture = new ImageInformation(this, "/resources/item_texture.dds");
        ImageInformation icon = new ImageInformation(this, "/resources/item_icon.dds");
        
        //Assign the model, texture and icon (we want to scale up the model, that can be done with the "modelSize" parameter)
        item.setModel(model, texture, 0.01f);
        item.setIcon(icon);
        
        //We don't want the item to be visible when holding it in the hands - so we select "None" as hand
        item.setHand(CustomItem.Hand.None);
        
        //Item can not be stacked (so max stack size is 1)
        item.setMaxStacksize(1);
        
        //Select a proper idle animation ("HoldPelt1" seems to be suitable)
        item.setPlayerIdleAnimation(Animation.Idle1);
        
        //Setup localized names. Since the game currently only supports English and German, we only set these names
        item.setLocalizedNames("en=3D-Model Blueprint", "de=3D-Modell Bauplan");
        
        //Now the important part: We set a custom secondary action, i.e. if the player
        //uses the right mouse button, this action is executed. If the player presses his
        //right mouse button, we check if there is already a model preview active
        //(we can just look if there is an attribute set). In this case, we place it,
        //otherwise we bring up the upload dialog
        item.setSecondaryAction(Animation.Idle1, 0f, (player, collision) -> {
            //Check if the model preview is active...
            if(player.hasAttribute(ATTRIBUTE_MODELPREVIEW)){
                //...then place the model in the world
                placeElement(player);
            }
            else{
                //...otherwise bring up file dialog to upload a model/texture
                showUploadDialog(player);
            }
        });
        
        //Once everything is set, register the custom item to the server
        getServer().registerCustomItem(item);
        
        //CUSTOM RECIPE
        
        //We want to be able to craft the item at the workbench, so we create a custom recipe for it
        CustomRecipe recipe = new CustomRecipe(CUSTOMITEM_UUID, CustomRecipe.Type.CustomItem, 0, "Miscellaneous", "workbench");
        
        //Model size for the preview (set a bigger preview size)
        recipe.setPreviewSize(0.0225f);
        
        //Set the ingredients required to craft this item
        recipe.setIngredients("16x goldingot");
        
        //Grab the item name
        recipe.setLocalizedNames(item.getLocalizedNames());
        
        //Set up localized descriptions (english [en] and german [de])
        recipe.setLocalizedDescriptions("en=This item allows you to upload \na custom 3D model from your hard drive \nand place it in the world!", "de=Ermöglicht dir, ein 3D Modell \nvon deiner Festplatte hochzuladen und \nin der Welt zu platzieren!");
        
        //Register the custom item to the server
        getServer().registerCustomRecipe(recipe);
        
        
        //Register this plugin as an event listener
        registerEventListener(this);
    }
    
    /**
     * A convenient function to create a new World3DModel element.
     * @param model the model.
     * @param texture the texture.
     * @param size the model size.
     * @param position the world position of the element.
     * @param rotation the rotation of the element.
     * @param collision true if it should have a collision shape, false if not
     * @return the newly created World3DModel element.
     */
    private World3DModel createNewWorld3DModel(ModelInformation model, ImageInformation texture, float size, Vector3f position, Quaternion rotation, boolean collision){
        World3DModel world3DModel = new World3DModel(model, texture, position, rotation);
        world3DModel.setScale(size);
        world3DModel.setLightingEnabled(true);
        world3DModel.setNpcCollisionEnabled(collision);
        world3DModel.setHittable(collision);
        world3DModel.setCollisionShape(collision ? CollisionShape.createHullCollisionShape() : null);
        world3DModel.setAttribute(ATTRIBUTE_STRENGTH, DEFAULT_ELEMENT_STRENGTH);
        return world3DModel;
    }
    
    @Override
    public void onDisable(){
        //If the database variable is set (not null), we close the database here
        if(database != null){
            database.close();
        }
    }
    
    @EventMethod
    public void onPlayerConnect(PlayerConnectEvent event){
        //Get the player who connected
        Player player = event.getPlayer();
        
        //Enable key listening, so we get notified if the player hits one of the registered keys
        player.setListenForKeyInput(true);
        
        //We only have to listen for certain keys (e.g. arrow keys)
        player.registerKeys(KeyInput.KEY_UP, KeyInput.KEY_DOWN, KeyInput.KEY_LEFT, KeyInput.KEY_RIGHT, KeyInput.KEY_PGUP, KeyInput.KEY_PGDN, KeyInput.KEY_ADD, KeyInput.KEY_SUBTRACT, KeyInput.KEY_ESCAPE);
        
        //Iterate through our world3DModels list and add every element from that list to our player
        for(World3DModel model : world3DModels){
            player.addWorldElement(model);
        }
        
        //Create a background panel and store it as an attribute (so we can access it later)
        GuiPanel background = new GuiPanel(0.5f, 0.5f, true, 1000, 650, false);
        background.setPivot(PivotPosition.Center);
        background.setColor(0f, 0f, 0f, 0.5f);
        background.setBorderThickness(10, false);
        background.setBorderColor(0f, 0f, 0f, 0.7f);
        background.setVisible(false);
        player.addGuiElement(background);
        player.setAttribute(ATTRIBUTE_FILEBROWSER_BACKGROUND, background);
        
        //Create a header which should appear slightly above our background panel. We want it to
        //be a child of our background panel. We also store it as an attribute (so we can access it later)
        GuiLabel header = new GuiLabel(0.5f, 1.1f, true);
        header.setPivot(PivotPosition.Center);
        header.setFontSize(48);
        header.setFont(Font.Default_Bold);
        header.setFontColor(1f, 1f, 1f, 1f);
        background.addChild(header);
        player.addGuiElement(header);
        player.setAttribute(ATTRIBUTE_FILEBROWSER_HEADLINE, header);
        
        //Last but not least create the actual file browser element. Add it as a child of the background
        //panel and store it as an attribute (so we can access it later)
        GuiFileBrowser fb = new GuiFileBrowser(14, 0.5f, 0.5f, true, 950, 600, false);
        fb.setPivot(PivotPosition.Center);
        fb.setPreviewImageEnabled(false);
        fb.setPreviewImageDimension(256);
        fb.setPreviewImagePosition(800, 350, false);
        background.addChild(fb);
        player.addGuiElement(fb);
        player.setAttribute(ATTRIBUTE_FILEBROWSER, fb);
    }
    
    /**
     * This method if called if the player changes the eqiupped item. The only thing
     * we want to do here is abort the model/texture upload progress (and hide the dialog,
     * if it is currently visible)
     * @param event the event.
     */
    @EventMethod
    public void onPlayerChangeItem(PlayerChangeEquippedItemEvent event){
        //Get the player who changed his item
        Player player = event.getPlayer();
        
        //If timer and preview attribute is set, model placing is currently active.
        //In this case, we want to abort it
        if(player.hasAttribute(ATTRIBUTE_TIMER) && player.hasAttribute(ATTRIBUTE_MODELPREVIEW)){
            abortPlacement(player);
        }
    }
    
    @EventMethod
    public void onPlayerSelectFileForUpload(PlayerSelectFileEvent event){
        //Get the player who selected a file for upload
        Player player = event.getPlayer();
        
        //Close the upload dialog
        hideUploadDialog(player);
        
        //Get the name of the file that was selected
        String filename = event.getFilename();
        
        //Check if such a file already exists
        File targetFile = new File(getPath() + "/Models/" + player.getUID() + "/" + filename);
        if(!targetFile.exists()){
            //Request a file upload from the client
            event.requestFileUpload(getPath() + "/Models/" + player.getUID(), (file) -> {
                //Once the upload is ready, call the "finalize" method
                finalizeUpload(player, file);
            });
        }
        else{
            //If the file already exists, call the "finalize" method
            finalizeUpload(player, targetFile);
        }
    }
    
    /**
     * This method brings up an upload dialog for the client (which either asks the client
     * to select and upload a texture or a model).
     * @param player the client we want to show the dialog for.
     */
    private void showUploadDialog(Player player){
        //Check if a model already exists
        if(player.hasAttribute(ATTRIBUTE_PENDING_MODEL)){
            //If a model exists, check if there is no texture - then we bring up the texture
            //upload dialog, otherwise we do nothing
            if(!player.hasAttribute(ATTRIBUTE_PENDING_TEXTURE)){
                //Get the header label and set its text
                GuiLabel header = (GuiLabel) player.getAttribute(ATTRIBUTE_FILEBROWSER_HEADLINE);
                header.setText("-TEXTURE- SELECTION");
                
                //Get the file browser element, set a proper file extension filter and enable the image preview
                GuiFileBrowser fb = (GuiFileBrowser) player.getAttribute(ATTRIBUTE_FILEBROWSER);
                fb.setFileExtensionFilter("jpg", "png", "dds");
                fb.setPreviewImageEnabled(true);
                
                //Get the background parent object and set it to visible
                GuiPanel background = (GuiPanel) player.getAttribute(ATTRIBUTE_FILEBROWSER_BACKGROUND);
                background.setVisible(true);
                
                //Remember to enable the mouse cursor
                player.setMouseCursorVisible(true);
            }
        }
        //If no model exists, bring up the model selection dialog
        else{
            //Get the header label and set its text
            GuiLabel header = (GuiLabel) player.getAttribute(ATTRIBUTE_FILEBROWSER_HEADLINE);
            header.setText("-MODEL- SELECTION");
            
            //Get the file browser element, set a proper file extension filter and disable the image preview
            GuiFileBrowser fb = (GuiFileBrowser) player.getAttribute(ATTRIBUTE_FILEBROWSER);
            fb.setFileExtensionFilter("obj", "fbx", "blend", "j3o");
            fb.setPreviewImageEnabled(false);
            
            //Get the background parent object and set it to visible
            GuiPanel background = (GuiPanel) player.getAttribute(ATTRIBUTE_FILEBROWSER_BACKGROUND);
            background.setVisible(true);
            
            //Also remember to enable the mouse cursor
            player.setMouseCursorVisible(true);
        }
    }
    
    /**
     * This method hides the upload dialog for a player.
     * @param player the client.
     */
    private void hideUploadDialog(Player player){
        //We just hide the background panel - since the file browser is a child of it,
        //it will no longer be visible then
        GuiPanel background = (GuiPanel) player.getAttribute(ATTRIBUTE_FILEBROWSER_BACKGROUND);
        background.setVisible(false);
        
        //Also disable the mouse cursor
        player.setMouseCursorVisible(false);
    }
    
    /**
     * Call this method once the upload is completed. This method also checks if
     * both a model and a texture are uploaded, and finalizes the process accordingly.
     * @param player the client who uploaded the files.
     * @param file the file which was just uploaded.
     */
    private void finalizeUpload(Player player, File file){
        try{
            //Get the file name (we use its extension to find out what type of file was uploaded)
            String filename = file.getName().toLowerCase();
            
            //If the file has a model format, it is obviously our model
            if(filename.endsWith("obj") || filename.endsWith("fbx") || filename.endsWith("j3o") || filename.endsWith("blend")){
                //Create a new model information object and save it as an attribute for the player
                ModelInformation model = new ModelInformation(file);
                player.setAttribute(ATTRIBUTE_PENDING_MODEL, model);
            }
            //If it has an image format, it is obviously our texture
            else if(filename.endsWith("jpg") || filename.endsWith("png") || filename.endsWith("dds")){
                //Create a new image information object and save it as an attribute for the player
                ImageInformation image = new ImageInformation(file);
                player.setAttribute(ATTRIBUTE_PENDING_TEXTURE, image);
            }
            
            //Check if both files (model and texture) are already uploaded
            if(player.hasAttribute(ATTRIBUTE_PENDING_MODEL) && player.hasAttribute(ATTRIBUTE_PENDING_TEXTURE)){
                //In that case, the upload process is ready
                onUploadCompleted(player);
                
                //Hide the upload dialog (it's no longer needed for this model)
                hideUploadDialog(player);
            }
            //Otherwise bring up the upload dialog again (so the client can upload the texture now)
            else{
                showUploadDialog(player);
            }
        }
        //This catch-block is called if an exception occurred
        catch(Exception e){
            //We print the stack trace, it's easier for debugging then
            e.printStackTrace();
            
            //Delete the model and texture attributes
            player.deleteAttribute(ATTRIBUTE_PENDING_MODEL);
            player.deleteAttribute(ATTRIBUTE_PENDING_TEXTURE);
            
            //Hide the upload dialog
            hideUploadDialog(player);
        }
    }
    
    /**
     * This method is called when the upload process is ready, i.e. if both a model
     * and a texture are uploaded. It creates a new World3DModel preview and allows
     * the player to place it in the world.
     * @param player the client.
     */
    private void onUploadCompleted(Player player){
        //Get the uploaded model and texture
        ModelInformation model = (ModelInformation) player.getAttribute(ATTRIBUTE_PENDING_MODEL);
        ImageInformation texture = (ImageInformation) player.getAttribute(ATTRIBUTE_PENDING_TEXTURE);
        
        //We don't need these attributes anymore, so we delete them
        player.deleteAttribute(ATTRIBUTE_PENDING_MODEL);
        player.deleteAttribute(ATTRIBUTE_PENDING_TEXTURE);

        //To get an initial position, we use our util function to get a position in front of the player
        Vector3f position = Utils.VectorUtils.getXYZInFrontOfPlayer(player, 2f);
        
        //When it comes to the initial rotation, we just create a new quaternion
        Quaternion rotation = new Quaternion();

        //We now create a new World3DModel with no collision (since it should serve as our placement preview)
        World3DModel world3DModel = createNewWorld3DModel(model, texture, 1f, position, rotation, false);
        
        //Make the preview slightly transparent
        world3DModel.setAlpha(0.9f);
        
        //Remember to enable transparency
        world3DModel.setTransparencyEnabled(true);
        
        //Add the world element to the players world
        player.addWorldElement(world3DModel);
        
        //Optional: We want our model to be properly sized, so we get the BoundingInformation
        //of the model file. It helps us to find out the initial size of the model. This size
        //can be used to scale the model up or down (so it always has a fixed initial size)
        BoundingInformation boundingInfo = model.getBoundingInformation();
        if(boundingInfo != null){
            //Get the X, Y and Z extents of the model
            float x = boundingInfo.getXExtent();
            float y = boundingInfo.getYExtent();
            float z = boundingInfo.getZExtent();
            float max;
            
            //We want to find out the biggest value, i.e. the biggest extent
            if(x > y) max = x > z ? x : z;
            else max = y > z ? y : z;
            
            //Only proceed if the biggest extent is greater than 0
            if(max > 0){
                //Scale the model (we want it to have a size of 2 blocks)
                world3DModel.setScale(2f / max);
            }
        }
        
        //Create a new timer which is responsible for our preview placement. We want
        //the timer to trigger every 50 milliseconds (0.05 seconds)
        Timer timer = new Timer(0.05f, 0.5f, -1, () -> {
            //Use a raycast to find a collision in front of the player
            player.raycast(COLLISION_BITMASK, (result) -> {
                //If a collision was found, we check if the distance is not greater than 6 units
                Vector3f targetPosition;
                if(result != null && result.hasCollision() && result.getDistance() <= 6f){
                    targetPosition = result.getCollisionPoint();
                }
                else{
                    targetPosition = Utils.VectorUtils.getXYZInFrontOfPlayer(player, 6f);
                    targetPosition.y = world3DModel.getPosition().y;
                }
                //Move the element to the target position
                world3DModel.moveTo(targetPosition, 9f);
            });
        });
        
        //Start the timer
        timer.start();

        //Store the timer and the model preview as an attribute
        player.setAttribute(ATTRIBUTE_TIMER, timer);
        player.setAttribute(ATTRIBUTE_MODELPREVIEW, world3DModel);
    }
    
    private void placeElement(Player player){
        //Get the timer which is used to determine a new position for placement
        //(we grab it from an attribute)
        Timer timer = (Timer) player.getAttribute(ATTRIBUTE_TIMER);
        
        //Get the model preview (we also get that from our attribute)
        World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);

        //Kill the timer (we don't need it anymore)
        timer.kill();

        //We just re-use our preview model to place it, but first we have change a few
        //settings, e.g. we set the alpha value back to 1 (fully opaque) and disable transparency...
        preview.setAlpha(1f);
        preview.setTransparencyEnabled(false);
        
        //...we also want npcs to collide with this element again...
        preview.setNpcCollisionEnabled(true);
        
        //...and it should be hittable...
        preview.setHittable(true);
        
        //...aaaand we want to set up a collision shape for the model (so players cannot walk through it)
        preview.setCollisionShape(CollisionShape.createHullCollisionShape());

        //Since the preview was only registered to the player who wanted to place it, we
        //have to register it to all other players now too
        for(Player p : getServer().getAllPlayers()){
            //Check if player is NOT our player (since we don't want to add the model twice for him)
            if(p.getID() != player.getID()){
                p.addWorldElement(preview);
            }
            
            //Play a proper "placement sound" for every player. We just use one of the game sounds for that
            p.playGameSound("place_block", preview.getPosition());
        }
        
        //Add the "World3DModel" to our list
        world3DModels.add(preview);
        
        //Show a small info text, so the player knows the model was placed sucessfully
        player.showStatusMessage("Model '"+preview.getModel().getFilename()+"' placed!", 2);
        
        //Last but not least we have to store the newly placed model in our database
        //(so we can retrieve it after a server restart or game restart). First we
        //recover the file names of the model and texture
        String modelFilename = preview.getModel().getFilename() + "." + preview.getModel().getExtension();
        String textureFilename = preview.getTexture().getFilename() + "." + preview.getTexture().getExtension();
        
        //Execute an INSERT statement to store all relevant information about the element
        database.executeUpdate("INSERT INTO `models` (`model`, `texture`, `playeruid`, `position`, `rotation`, `size`, `collision`) VALUES ('" + modelFilename + "', '" + textureFilename + "', '" + player.getUID() + "', '" + preview.getPosition().toString() + "', '" + preview.getRotation().toString() + "', '" + preview.getScale().y + "', '" + (byte)(preview.hasCollisionShape() ? 1 : 0) + "')");
        
        //Delete the "timer" and "preview" attributes from the player (since we don' need them anymore)
        player.deleteAttribute(ATTRIBUTE_TIMER);
        player.deleteAttribute(ATTRIBUTE_MODELPREVIEW);
    }
    
    /**
     * Aborts the model placing process. This removes the timer and the preview and deletes
     * the related player attributes.
     * @param player the client.
     */
    private void abortPlacement(Player player){
        //Get the timer which is responsible for placement (which is stored as an attribute)
        Timer timer = (Timer) player.getAttribute(ATTRIBUTE_TIMER);
        
        //Get the model preview (we also get that from our attribute)
        World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);

        //Kill the timer
        timer.kill();
        
        //Remove the world element from the player
        player.removeWorldElement(preview);

        //Delete the "timer" and "preview" attributes from the player
        player.deleteAttribute(ATTRIBUTE_TIMER);
        player.deleteAttribute(ATTRIBUTE_MODELPREVIEW);
    }
    
    /**
     * This event is triggered when the player presses a key (only works for keys which
     * were registered to the client  previously, see the "onPlayerConnect()" event
     * @param event the key input event.
     */
    @EventMethod
    public void onPlayerKeyInput(PlayerKeyEvent event){
        //Get the player who pressed the key
        Player player = event.getPlayer();
        
        //We only want to process "key down" events, i.e. when the key is pressed
        if(event.isPressed()){
            //Check if model placing is currently active: in this case, both attributes are set
            if(player.hasAttribute(ATTRIBUTE_TIMER) && player.hasAttribute(ATTRIBUTE_MODELPREVIEW)){
                //It's  more convenient to store our "rotation step" (i.e. how much we want to rotate the model per button press) in a final variable
                final float ROTATION_STEP = Utils.MathUtils.QUARTER_PI * 0.25f;
                
                //If pressed key is the left arrow key, we rotate around the Y axis
                if(event.getKeyCode() == KeyInput.KEY_LEFT){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(0f, ROTATION_STEP, 0f));
                    preview.setRotation(rot);
                }
                //If pressed key is the right arrow key, we rotate around the Y axis (negative direction)
                else if(event.getKeyCode() == KeyInput.KEY_RIGHT){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(0f, -ROTATION_STEP, 0f));
                    preview.setRotation(rot);
                }
                //If pressed key is the up arrow key, we rotate around the X axis
                else if(event.getKeyCode() == KeyInput.KEY_UP){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(ROTATION_STEP, 0f, 0f));
                    preview.setRotation(rot);
                }
                //If pressed key is the down arrow key, we rotate around the X axis (negative direction)
                else if(event.getKeyCode() == KeyInput.KEY_DOWN){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(-ROTATION_STEP, 0f, 0f));
                    preview.setRotation(rot);
                }
                //If pressed key is the page-up key, we rotate around the Z axis
                else if(event.getKeyCode() == KeyInput.KEY_PGUP){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(0f, 0f, ROTATION_STEP));
                    preview.setRotation(rot);
                }
                //If pressed key is the page-down key, we rotate around the Z axis (negative direction)
                else if(event.getKeyCode() == KeyInput.KEY_PGDN){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    Quaternion rot = new Quaternion(preview.getRotation()).multLocal(new Quaternion().fromAngles(0f, 0f, -ROTATION_STEP));
                    preview.setRotation(rot);
                }
                //If pressed key is the + key, we want to scale the model up
                else if(event.getKeyCode() == KeyInput.KEY_ADD){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    preview.setScale(preview.getScale().x * 1.1f);
                }
                //If pressed key is the - key, we want to scale the model down
                else if(event.getKeyCode() == KeyInput.KEY_SUBTRACT){
                    World3DModel preview = (World3DModel) player.getAttribute(ATTRIBUTE_MODELPREVIEW);
                    preview.setScale(preview.getScale().x * 0.9f);
                }
                //If key is ESC key, we want to abort the model placing
                else if(event.getKeyCode() == KeyInput.KEY_ESCAPE){
                    abortPlacement(player);
                }
            }
            else{
                //We have to check for the ESC key again: if model placing isn't currently active,
                //there is possibly the upload dialog visible. In this case, we hide it and abort
                //the uploading process
                if(event.getKeyCode() == KeyInput.KEY_ESCAPE){
                    hideUploadDialog(player);
                }
            }
        }
    }
    
    /**
     * Called when the world element is hit by an item. We use this event to enable
     * the player to destroy the element.
     * @param event the event.
     */
    @EventMethod
    public void onPlayerHitWorldElement(PlayerElementHitEvent event){
        //Get the player who hit the object
        Player player = event.getPlayer();
        
        //Get the currently equipped item of the player (which should be the item the player used to hit the object)
        Item equippedItem = player.getEquippedItem();
        
        //Just to avoid any NPEs, check if the item is not null
        if(equippedItem != null){
            //Get the damage definition for this item
            Definitions.HitDamageDefinition damageDef = Definitions.getHitDamageDefinition(equippedItem.getName());
            
            //Check if the item has a damage definition and if the "object damage" is greater than 0
            if(damageDef != null && damageDef.getObjectDamage() > 0){
                //Get the world element which was hit
                WorldElement model = event.getWorldElement();
                
                //Check if it has the strengh attribute
                if(model.hasAttribute(ATTRIBUTE_STRENGTH)){
                    //Get the strength value...
                    int strength = (int) model.getAttribute(ATTRIBUTE_STRENGTH);
                    
                    //...and reduce it according to the damage definition
                    strength -= damageDef.getObjectDamage();

                    //Update the strength attribute
                    model.setAttribute(ATTRIBUTE_STRENGTH, strength);
                    
                    //If strength is now <= 0, we destroy the element
                    if(strength <= 0){
                        //Iterate through all players and remove the element for them
                        for(Player p : getServer().getAllPlayers()){
                            p.removeWorldElement(model);
                            
                            //Play a proper destroy sound 
                            p.playGameSound("item_break_stone_medium", model.getPosition());
                        }
                        
                        //Now remove the element from the database
                        database.executeUpdate("DELETE FROM `models` WHERE `position` = '"+model.getPosition().toString()+"'");
                        
                        //Finally remove it from our internal "world3DModels" list
                        world3DModels.remove((World3DModel) model);
                    }
                }
            }
        }
    }
    
    /**
     * Called when a player crafts a new item. We want to make sure that only admins
     * are able to craft our custom item, so we cancel the event for all other players.
     * @param event the item craft event.
     */
    @EventMethod
    public void onItemCraft(PlayerCraftItemEvent event){
        //Get the player who wants to craft the item
        Player player = event.getPlayer();
        
        //Get the item the player is about to craft
        Item item = event.getItem();
        
        //Check if the item attribute is a "CustomItemAttribute" (then it's definitely a custom item)
        if(item != null && item.getAttribute() instanceof Item.CustomItemAttribute){
            //Cast the attribute to a CustomItemAttribute
            Item.CustomItemAttribute attribute = (Item.CustomItemAttribute) item.getAttribute();
            
            //Check if the item has the same UUID as our custom item (see above)
            if(attribute.getUUID().equals(CUSTOMITEM_UUID)){
                //If player is not an admin, cancel the event. This means only admins are able to craft this item
                if(!player.isAdmin()){
                    event.setCancelled(true);
                    
                    //Send a small notification to the player
                    player.showStatusMessage("You are not allowed to craft this item!", 4);
                }
            }
        }
    }
    
}
