package net.risingworld.api.example.customitem.gun;

import net.risingworld.api.Plugin;
import net.risingworld.api.objects.Player;
import net.risingworld.api.objects.WorldItem;
import net.risingworld.api.objects.custom.CustomItem;
import net.risingworld.api.objects.custom.CustomRecipe;
import net.risingworld.api.utils.Animation;
import net.risingworld.api.utils.Definitions;
import net.risingworld.api.utils.ImageInformation;
import net.risingworld.api.utils.ModelInformation;
import net.risingworld.api.utils.Quaternion;
import net.risingworld.api.utils.Utils;
import net.risingworld.api.utils.Vector3f;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * 
 * @author red51
 */

public class PotatoCannon extends Plugin{
    
    //Final variable (constant) to define the UUID of the item (has to be unique!)
    public final String GUN_UUID = "net.risingworld.api.example.customitem.potatocannon";
    
    @Override
    public void onEnable(){
        //Create our new custom potato cannon item, make sure to use a unique ID (first parameter)
        CustomItem item = new CustomItem(GUN_UUID, "potatocannon");
        
        //Load the model, texture and icon
        ModelInformation model = new ModelInformation(getPath() + "/assets/potatocannon.obj");
        ImageInformation texture = new ImageInformation(getPath() + "/assets/potatocannon.dds");
        ImageInformation icon = new ImageInformation(getPath() + "/assets/potatocannon_icon.png");
        
        //Assign the model, texture and icon (we want to scale up the model, that can be done with the "modelSize" parameter)
        item.setModel(model, texture, 2.15f);
        item.setIcon(icon);
        
        //Item can not be stacked (so max stack size is 1)
        item.setMaxStacksize(1);
        
        //Select a proper idle animation ("M14Aim" seems to be suitable)
        item.setPlayerIdleAnimation(Animation.M14Aim);
        
        //Set a suitable item position, rotation and first person body position / rotation.
        //This part is a little bit tricky: To get proper coordinates, you can use the console
        //command "debugitem" to change the item position with your arrow keys (hold SHIFT to
        //rotate it). Press RETURN to copy the coordinates to clipboard (press SHIFT+RETURN
        //to copy the rotation).
        //For the first person body position, use the command "debugplayerbody" 
        //(which just works like the debugitem-command)
        item.setItemPosition(-0.2210794f, 0.09963215f, 0.30302262f);
        item.setItemRotation(0.23054627f, -0.43278024f, -0.5497708f, 0.6762362f);
        item.setFPBodyPosition(-0.39670023f, -0.5547194f, 0.261761f);
        item.setFPBodyRotation(-0.04698471f, 0.018427098f, -0.066861615f, 0.99649376f);
        
        //Setup localized names. Since the game currently only supports English and German, we only set these languages
        item.setLocalizedNames("en=Potato Cannon", "de=Kartoffelkanone");
        
        //Now the important part: We set a custom action, i.e. if the player
        //uses the left mouse button, this action is executed. In this case, we want
        //to spawn and shoot a potato in front of the player
        item.setPrimaryAction(Animation.M14Fire, 0.1f, (player, collision) -> {
            //To keep the code more structured, we've created a separate "shootProjectile()" method
            //which handles the whole shooting process
            shootProjectile(player);
        });
        
        //Once everything is set, register the custom item to the server
        getServer().registerCustomItem(item);
        
        
        //CUSTOM RECIPE
        
        //We want to be able to craft the item at the workbench, so we create a CustomRecipe for it
        CustomRecipe recipe = new CustomRecipe(GUN_UUID, CustomRecipe.Type.CustomItem, 0, "Miscellaneous", "workbench");
        
        //Preview model size. In this case, we use the same size as the model size (just slightly smaller)
        recipe.setPreviewSize(item.getModelSize() * 0.9f);
        
        //Set the ingredients required to craft this item
        recipe.setIngredients("16x ironplate", "16x goldplate");
        
        //For the name of the item, we just grab it from our actual custom item
        recipe.setLocalizedNames(item.getLocalizedNames());
        
        //Set up a description
        recipe.setLocalizedDescriptions("...");
        
        //Register the custom item to the server
        getServer().registerCustomRecipe(recipe);
    }
    
    @Override
    public void onDisable(){
        //We don't need this function, but we still have to override it
    }
    
    private void shootProjectile(Player player){
        //We have to find out the weapon world position, or more precisely, the muzzle position.
        //We also want to define a shooting direction. 
        Vector3f weaponPosition = new Vector3f(-0.25f, 0.0f, 1.0f);
        Vector3f shootingDirection = new Vector3f(0f, 0.15f, 0.85f).normalizeLocal();

        //It's important to rotate our muzzle position and shooting direction in accordance
        //to the player rotation. To apply a rotation to a vector, we have to multiply
        //this vector with the quaternion, i.e. "rotation.multLocal(vector)" (this stores
        //the final values in the vector).
        player.getRotation().multLocal(weaponPosition);
        player.getRotation().multLocal(shootingDirection);

        //Now add the player world position to the "weaponPosition" offset to get
        //the final world position of the weapon/muzzle
        weaponPosition.addLocal(player.getPosition()).addLocal(0f, 1.35f, 0f);

        //Get a proper rotation for the potato (so it looks like it was fired from the cannon)
        Quaternion potatoRotation = new Quaternion().lookAt(shootingDirection);
        potatoRotation.multLocal(new Quaternion().fromAngles(0f, Utils.MathUtils.HALF_PI, 0f));

        //Get the item definition of a potato
        Definitions.ItemDefinition itemDef = Definitions.getItemDefinition("potato");

        //Spawn a new potato (our "projectile") in the world (make sure it make it "physical")
        WorldItem projectile = getWorld().spawnItem(itemDef.getID(), 0, 1, weaponPosition, potatoRotation, true);

        //Apply an impulse to the potato. We use the shooting direction and multiply
        //it with a factor of 15.0f
        projectile.applyPhysicalImpulse(shootingDirection.mult(15f));

        //Play a proper shooting sound. In this case, "hit_pumpkin" is more or less suitable,
        //so we don't have to create our own custom sound
        player.playGameSound("hit_pumpkin", weaponPosition);
    }
    
}
