package net.risingworld.api.example.customitems;

import java.io.File;
import net.risingworld.api.Plugin;
import net.risingworld.api.objects.custom.CustomItem;
import net.risingworld.api.objects.custom.CustomRecipe;
import net.risingworld.api.utils.Animation;
import net.risingworld.api.utils.Definitions;
import net.risingworld.api.utils.ImageInformation;
import net.risingworld.api.utils.ModelInformation;
import net.risingworld.api.utils.Quaternion;
import net.risingworld.api.utils.Utils;
import net.risingworld.api.utils.Vector3f;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).<br>
 * 
 * @author red51
 */

public class CustomItemLoader extends Plugin{
    
    /**
     * This method is called when the plugin is loaded / initialized. We use it to search all
     * music tracks that can be used by our plugin
     */
    @Override
    public void onEnable(){
        //We check our subfolder "Items" and read all subfolders
        File dir = new File(getPath() + "/Items/");
        
        //If dir does not exist, we create it
        if(!dir.exists()){
            dir.mkdir();
        }
        //Otherwise we check all subdirectories
        else{
            //Get an array containing all subfiles and -folders
            File[] subfolders = dir.listFiles();
            
            //Iterate through all subfolders/-files
            for(File subfolder : subfolders){
                //Check if file is a directory, otherwise we ignore it
                if(subfolder.isDirectory()){
                    //We create a try-catch-block, so if an exception occurs while loading a custom
                    //item, the game will not crash (instead the "broken" item will just be skipped)
                    try{
                        //Check if "ItemDef" could be loaded
                        ItemDef itemDef = loadCustomItemDefinition(subfolder);

                        //If the file exists, we use it to create a custom item
                        if(itemDef != null){
                            createCustomItem(itemDef);
                        }
                    }
                    catch(Exception e){
                        //Print an error message (which shows up in the output log)
                        System.err.println("Unable to load item from folder '" + subfolder.getName() + "'");
                        //Print the stack trace, this is helpful to find out in which line the error occurred
                        e.printStackTrace();
                    }
                }
            }
        }
    }
    
    @Override
    public void onDisable(){
        //...
    }
    
    /**
     * This method creates and registers a custom item as well as a custom recipe.
     * @param itemDef our item definition class (see below) which contains all relevant information.
     */
    private void createCustomItem(ItemDef itemDef){
        //Just a debug output (so we can see in the log which item was created)
        System.out.println("CREATE CUSTOM ITEM: "+itemDef.name+"  ("+itemDef.uuid+")");
        
        //Create our new custom item, make sure to use a unique ID (first parameter)
        CustomItem item = new CustomItem(itemDef.uuid, itemDef.name);
        
        //Load the model, texture and icon
        ModelInformation model = new ModelInformation(itemDef.model);
        ImageInformation texture = new ImageInformation(itemDef.texture);
        ImageInformation icon = new ImageInformation(itemDef.icon);
        item.setModel(model, texture, itemDef.size);
        item.setIcon(icon);
        
        //Item can not be stacked
        item.setMaxStacksize(itemDef.stacksize);
        
        //Select a proper idle animation
        item.setPlayerIdleAnimation(itemDef.idleAnimation);
        
        //Set the "hand", i.e. define in which hand this item is held 
        //(left, right or "none" if the item should not be visible)
        item.setHand(itemDef.hand);
        
        //Item position and rotation (in hand)
        item.setItemPosition(itemDef.itemPosition);
        item.setItemRotation(itemDef.itemRotation);
        
        //First person body position and rotation
        item.setFPBodyPosition(itemDef.bodyPosition);
        item.setFPBodyRotation(itemDef.bodyRotation);
        
        //Primary Action Animation
        if(itemDef.primaryAnimation != null){
            item.setPrimaryAction(itemDef.primaryAnimation, itemDef.primaryTriggerDelay, null);
        }
        
        //Secondary Action Animation
        if(itemDef.secondaryAnimation != null){
            item.setSecondaryAction(itemDef.secondaryAnimation, itemDef.secondaryTriggerDelay, null);
        }
        
        //If a hit definition is set, we assign it (as well as the damage factor)
        if(itemDef.hitDefinition != null){
            //Get the hit definition
            Definitions.HitDamageDefinition damageDef = Definitions.getHitDamageDefinition(itemDef.hitDefinition);
            
            //Check if a proper hit definition name was set, otherwise we get a NPE
            if(damageDef != null){
                //Set the hit definition
                item.setHitDamageDefinition(damageDef);
                
                //Set a damage factor (default is 1)
                item.setHitDamageFactor(itemDef.hitDefinitionFactor);
            }
        }

        //Setup a name which will be displayed in the inventory
        item.setLocalizedNames(itemDef.name);
        
        //Register the custom item to the server
        getServer().registerCustomItem(item);
        
        
        //RECIPE
        
        //If a "category" is set, we will register a custom recipe for this item
        if(itemDef.category != null){
            //Create a new CustomRecipe object. We use the UUID of the item and the
            //(lower case) category and crafting station from the item definition
            CustomRecipe recipe = new CustomRecipe(itemDef.uuid, CustomRecipe.Type.CustomItem, 0, itemDef.category, itemDef.craftingStation);
            
            //Set a proper preview size (same value as item model size)
            recipe.setPreviewSize(item.getModelSize());
            
            //Set the ingredients which are required to craft this item
            recipe.setIngredients(itemDef.ingredients);

            //Grab the name (will be visible in crafting menu) from the item
            recipe.setLocalizedNames(item.getName());

            //Set up a description
            recipe.setLocalizedDescriptions(itemDef.description);

            //Register the custom item to the server
            getServer().registerCustomRecipe(recipe);
        }
    }
    
    
    //We define some names for the individual item information. This makes life 
    //easier when reading the item information from a text file
    
    private final String UUID = "uuid: ";
    private final String NAME = "name: ";
    private final String MODEL = "model: ";
    private final String TEXTURE = "texture: ";
    private final String ICON = "icon: ";
    private final String SIZE = "size: ";
    private final String HAND = "hand: ";
    private final String HITDEF = "hitdef: ";
    private final String HITFACTOR = "hitfactor: ";
    private final String STACKSIZE = "stack: ";
    private final String IDLEANIMATION = "idleanimation: ";
    private final String PRIMARYANIMATION = "primaryanimation: ";
    private final String SECONDARYANIMATION = "secondaryanimation: ";
    private final String PRIMARYTRIGGERDELAY = "primarytriggerdelay: ";
    private final String SECONDARYTRIGGERDELAY = "secondarytriggerdelay: ";
    private final String ITEMPOSITION = "itemposition: ";
    private final String ITEMROTATION = "itemrotation: ";
    private final String BODYPOSITION = "bodyposition: ";
    private final String BODYROTATION = "bodyrotation: ";
    private final String DESCRIPTION = "description: ";
    private final String INGREDIENTS = "ingredients: ";
    private final String CRAFTINGSTATION = "craftingstation: ";
    private final String CATEGORY = "category: ";
    
    /**
     * This function reads in a text file which contains all item information and 
     * store them in an "ItemDef" object.
     * @param folder the folder which contains the item information text file as 
     * well as the related assets (model, texture, icon etc).
     * @return an "ItemDef" object which contains the item information, or null
     * if there was no definition file.
     */
    private ItemDef loadCustomItemDefinition(File folder){
        //We use our convenient "findFile()" function to find the "ItemDefinition.txt" file in the folder
        File definition = findFile("ItemDefinition.txt", folder);
        
        //Chech if the definition was found
        if(definition != null){
            //Read the whole content of the file
            String content = Utils.FileUtils.readStringFromFile(definition);
            
            //Split by line breaks. We use a regex to cover both Windows and Linux line breaks.
            //This function returns an array of strings where every index contains a single line
            String[] lines = content.split("\r\n|\n|\r");  //we have to cover all kinds of line separators
            
            //Create a new empty ItemDef object
            ItemDef def = new ItemDef();
            
            //Now iterate through all lines and check them
            for(String line : lines){
                //We don't want to care about lower and upper case, so we just convert the line to lower case
                String l = line.toLowerCase();
                if(l.startsWith(UUID)){
                    def.uuid = line.substring(UUID.length());
                }
                else if(l.startsWith(NAME)){
                    def.name = line.substring(NAME.length());
                }
                else if(l.startsWith(MODEL)){
                    def.model = new File(folder, line.substring(MODEL.length()));
                }
                else if(l.startsWith(TEXTURE)){
                    def.texture = new File(folder, line.substring(TEXTURE.length()));
                }
                else if(l.startsWith(ICON)){
                    def.icon = new File(folder, line.substring(ICON.length()));
                }
                else if(l.startsWith(HITDEF)){
                    def.hitDefinition = line.substring(HITDEF.length());
                }
                else if(l.startsWith(HITFACTOR)){
                    String s = line.substring(HITFACTOR.length());
                    if(Utils.StringUtils.isNumeric(s)){
                        def.hitDefinitionFactor = Float.parseFloat(s);
                    }
                }
                else if(l.startsWith(SIZE)){
                    String s = line.substring(SIZE.length());
                    if(Utils.StringUtils.isNumeric(s)){
                        def.size = Float.parseFloat(s);
                    }
                }
                else if(l.startsWith(STACKSIZE)){
                    String s = line.substring(STACKSIZE.length());
                    if(Utils.StringUtils.isInteger(s)){
                        def.stacksize = Integer.parseInt(s);
                    }
                }
                else if(l.startsWith(HAND)){
                    String s = line.substring(HAND.length());
                    if(s.equalsIgnoreCase("left")) def.hand = CustomItem.Hand.Left;
                    else if(s.equalsIgnoreCase("right")) def.hand = CustomItem.Hand.Right;
                    else def.hand = CustomItem.Hand.None;
                }
                else if(l.startsWith(IDLEANIMATION)){
                    def.idleAnimation = Animation.valueOf(line.substring(IDLEANIMATION.length()));
                }
                else if(l.startsWith(PRIMARYANIMATION)){
                    def.primaryAnimation = Animation.valueOf(line.substring(PRIMARYANIMATION.length()));
                }
                else if(l.startsWith(SECONDARYANIMATION)){
                    def.secondaryAnimation = Animation.valueOf(line.substring(SECONDARYANIMATION.length()));
                }
                else if(l.startsWith(PRIMARYTRIGGERDELAY)){
                    String s = line.substring(PRIMARYTRIGGERDELAY.length());
                    if(Utils.StringUtils.isNumeric(s)){
                        def.primaryTriggerDelay = Float.parseFloat(s);
                    }
                }
                else if(l.startsWith(SECONDARYTRIGGERDELAY)){
                    String s = line.substring(SECONDARYTRIGGERDELAY.length());
                    if(Utils.StringUtils.isNumeric(s)){
                        def.secondaryTriggerDelay = Float.parseFloat(s);
                    }
                }
                else if(l.startsWith(ITEMPOSITION)){
                    String s = line.substring(ITEMPOSITION.length());
                    def.itemPosition.fromString(s);
                }
                else if(l.startsWith(ITEMROTATION)){
                    String s = line.substring(ITEMROTATION.length());
                    def.itemRotation.fromString(s);
                }
                else if(l.startsWith(BODYPOSITION)){
                    String s = line.substring(BODYPOSITION.length());
                    def.bodyPosition.fromString(s);
                }
                else if(l.startsWith(BODYROTATION)){
                    String s = line.substring(BODYROTATION.length());
                    def.bodyRotation.fromString(s);
                }
                else if(l.startsWith(DESCRIPTION)){
                    def.description = line.substring(DESCRIPTION.length());
                }
                else if(l.startsWith(CATEGORY)){
                    def.category = line.substring(CATEGORY.length());
                }
                else if(l.startsWith(CRAFTINGSTATION)){
                    def.craftingStation = line.substring(CRAFTINGSTATION.length()).toLowerCase();
                }
                else if(l.startsWith(INGREDIENTS)){
                    String s = Utils.StringUtils.removeAllWhitespaces(line.substring(INGREDIENTS.length())).toLowerCase();
                    if(s.contains(",")){
                        def.ingredients = s.split(",");
                    }
                    else{
                        def.ingredients = new String[]{s};
                    }
                }
            }
            
            //Now check if all relevant data was available
            if(def.uuid == null || def.uuid.isEmpty()){
                throw new IllegalStateException("UUID is not set!");
            }
            else if(def.name == null || def.name.isEmpty()){
                throw new IllegalStateException("Item name is not set!");
            }
            else if(def.model == null || !def.model.exists()){
                throw new IllegalStateException("Model info does not exist!");
            }
            else if(def.texture == null || !def.texture.exists()){
                throw new IllegalStateException("Texture info does not exist!");
            }
            else if(def.icon == null || !def.icon.exists()){
                throw new IllegalStateException("Icon info does not exist!");
            }
            
            //Return the item def
            return def;
        }
        return null;
    }
    
    /**
     * This function looks for a certain file in a certain directory.
     * @param filename the name of the file you're looking for.
     * @param parent the directory (parent directory of the file in question).
     * @return the file you're looking for, or null if the file was not found.
     */
    private File findFile(String filename, File parent){
        for(File f : parent.listFiles()){
            if(!f.isDirectory() && f.getName().equalsIgnoreCase(filename)){
                return f;
            }
        }
        return null;
    }
    
    
    /**
     * This is a helper class which allows us to store all relevant information
     * which are required to create and regsiter a custom item and a custom recipe.
     * Since we read all information from a text file, it's more convenient to
     * use a custom class to store them. 
     */
    private class ItemDef{
        
        String uuid;
        String name;
        
        File model;
        File texture;
        File icon;
        
        float size = 1.0f;
        
        String hitDefinition;
        float hitDefinitionFactor = 1.0f;
        
        int stacksize = 1;
        
        CustomItem.Hand hand = CustomItem.Hand.Right;
        
        Animation idleAnimation = Animation.Weapon1HandIdle;
        Animation primaryAnimation = Animation.Weapon1HandHit3;
        Animation secondaryAnimation = null;
        
        float primaryTriggerDelay = 0.5f;
        float secondaryTriggerDelay = 0.5f;
        
        Vector3f itemPosition = new Vector3f();
        Quaternion itemRotation = new Quaternion();
        Vector3f bodyPosition = new Vector3f();
        Quaternion bodyRotation = new Quaternion();
        
        //Crafting
        String category;
        String description = "";
        String craftingStation;
        String[] ingredients;
        
    }
    
}
