package net.risingworld.api.example.audio;

import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.SoundInformation;
import net.risingworld.api.utils.Vector3f;

/**
 * This is our "MusicPlayer" class. It contains all information about the currently
 * playing music track and has a convenient "play()" and "stop()" method, which allows
 * us to play/stop a music track for all players easily.
 * 
 * @author red51
 */

public class MusicPlayer{
        
    //A reference to the sign object
    private int id;
 
    //This vector contains the world position of this music player 
    //(actually the world position of the object this music player is assigned to)
    private Vector3f position;
    
    //A reference to our main plugin class - we need it to access its music tracks list,
    //but also to get the "Server" object in order to get access to all players
    private Jukebox plugin;

    //The track that's currently playing
    private int currentTrack = -1;

    //A timestamp (millis) when the track was started
    private long playStartTime = 0L;

    //Is the music player currently playing?
    private boolean isPlaying = false;
    
    //This is the default volume when playing music tracks. It's a public static final
    //variable, which means it cannot be changed and it can be accessed anywhere
    public static final float DEFAULT_VOLUME = 0.25f;
    
    //Min range where the music track can still be heared at 100% volume
    public static final float MIN_RANGE = 1f;
    
    //Max range where the music track can be heared
    public static final float MAX_RANGE = 25f;

    //This is the name of the attribute where we store the sound ID per player.
    //We put it into a static final variable so we don't have to use the actual
    //string again and again (we're using the "private" modifier since we don't want
    //anyone from outside to access this variable)
    private static final String TRACK_ATTRIBUTE = "net.risingworld.api.jukebox.track";
    
    /**
     * Creates a new music player. 
     * @param plugin our main plugin class (the "Jukebox" plugin).
     * @param id the ID of the related object, i.e. the sign ID.
     * @param position the world position of the music player.
     */
    public MusicPlayer(Jukebox plugin, int id, Vector3f position){
        this.id = id;
        this.plugin = plugin;
        this.position = new Vector3f(position);
    }
    
    /**
     * Gets whether or not the music player is currently playing.
     * @return true if the music player is playing, false if not.
     */
    public boolean isPlaying(){
        return isPlaying;
    }
    
    /**
     * Gets the current track (which is playing).
     * @return the current track ID, or -1 if no track is playing.
     */
    public int getCurrentTrack(){
        return currentTrack;
    }
    
    /**
     * Gets the world position of the music player.
     * @return the world coordinates (XYZ), as a Vector3f.
     */
    public Vector3f getPosition(){
        return position;
    }
    
    /**
     * Starts playing a music track (only works if this track exists). This method
     * plays the music track for all players who are currently connected to the server.
     * @param track the ID of the track we want to play.
     */
    public void play(int track){
        //Check if the track does actually exists
        if(track >= 0 && track < plugin.musicTracks.size()){
            //Assign the track ID to our internal "currentTrack" variable
            currentTrack = track;
            
            //Get the start time (the current timestamp in milliseconds)
            playStartTime = System.currentTimeMillis();
            
            //Mark this player as "playing"
            isPlaying = true;
            
            //Get the SoundInformation which belongs to the music track
            SoundInformation musicTrack = plugin.musicTracks.get(track);
            
            //Now we want play the sound for all players and store their individual soundID as an attribute.
            //Since our music player has a special attribute name format, we created a separate function and use
            //it to get the name for the attribute (this is easier than writing the whole attribute name manually 
            //every time)
            String attribute = getAttributeName();
            
            //Get a list of all players and iterate through it
            for(Player player : plugin.getServer().getAllPlayers()){
                //Check if the player is not null and if he is connected
                if(player != null && player.isConnected()){
                    //Now play the music track as a loop, with given parameters (volume, range etc). This
                    //function returns a sound ID, which can be used to reference this particular sound on the client.
                    //We need the sound ID later if we want to stop the sound
                    int soundID = player.playSound(musicTrack, true, DEFAULT_VOLUME, 1f, MIN_RANGE, MAX_RANGE, position);
                    
                    //Store the sound ID as attribute for the player
                    player.setAttribute(attribute, soundID);
                }
            }
        }
    }
    
    /**
     * Stops the music player, i.e. if checks if the track is playing for any players
     * and stops it accordingly.
     */
    public void stop(){
        //Get the attribute name which was used to store the sound ID (when the music track started playing)
        String attribute = getAttributeName();
        
        //Get a list of all players and iterate through it
        for(Player player : plugin.getServer().getAllPlayers()){
            //Check if the player is not null and if he is connected
            if(player != null && player.isConnected()){
                //Check if the player actually has such an attribute
                if(player.hasAttribute(attribute)){
                    //In that case get the attribute and cast it to an int
                    int soundID = (int) player.getAttribute(attribute);
                    
                    //Use that ID to stop the sound for the player
                    player.stopSound(soundID);
                    
                    //Finally delete the attribute (since we don't need that sound ID anymore)
                    player.deleteAttribute(attribute);
                }
            }
        }
        
        //Set the current track to -1 and mark this music player as "not playing"
        currentTrack = -1;
        isPlaying = false;
    }
    
    /**
     * Gets a (millisecond) timestamp when the last track was played. This is used for new players (who just joined the game)
     * so we can keep the timeline position of the track in sync with other players.
     * @return the timestamp when the track was started.
     */
    public long getPlayStartTime(){
        return playStartTime;
    }
    
    /**
     * Gets the name of the attribute where we store the sound ID for the player. It has a special
     * prefix (hoping not to collide with other plugins) and contains the music player ID and the current track ID.
     * @return an attribute name which can be used to store the sound ID for a player.
     */
    public String getAttributeName(){
        return TRACK_ATTRIBUTE + "_" + id + "_" + currentTrack;
    }

}

