package net.risingworld.api.example.audio;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import net.risingworld.api.Plugin;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.Threading;
import net.risingworld.api.events.player.PlayerObjectInteractionEvent;
import net.risingworld.api.events.player.PlayerSpawnEvent;
import net.risingworld.api.events.player.world.PlayerDestroyObjectEvent;
import net.risingworld.api.events.player.world.PlayerRemoveObjectEvent;
import net.risingworld.api.objects.Player;
import net.risingworld.api.objects.Sign;
import net.risingworld.api.utils.Definitions;
import net.risingworld.api.utils.SoundInformation;
import net.risingworld.api.utils.Utils;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * 
 * @author red51
 */

public class Jukebox extends Plugin implements Listener{
    
    //An ArrayList containing all music tracks which were found in our music folder.
    //We use a "protected" modifier so our "MusicPlayer" class (which is in the same package)
    //can access this list
    protected ArrayList<SoundInformation> musicTracks = new ArrayList<>();
    
    //An array containing all music track names (used for the context menu which is shown to the player).
    //Since no other class needs to access this variable, we can use a "private" modifier (although that doesn't really matter here)
    private String[] musicTrackNames;
    
    //A HashMap containing all music players and their according ID (mapping ID -> music player)
    private HashMap<Integer, MusicPlayer> musicPlayers = new HashMap<>();
    
    /**
     * This method is called when the plugin is loaded / initialized. We use it to search all
     * music tracks that can be used by our plugin. In addition to that, we want our plugin to
     * listen for "events" (e.g. when a player spawns, an object gets destroyed etc), so we register
     * our plugin as "EventListener" at the bottom of this method.
     */
    @Override
    public void onEnable(){
        //Get the music folder
        File musicFolder = getMusicFolder();
        
        //If the directory exists, we will read all files from it
        if(musicFolder != null && musicFolder.exists() && musicFolder.isDirectory()){
            File[] files = musicFolder.listFiles();
            for(File f : files){
                //We will ignore directories inside our music folder
                if(!f.isDirectory()){
                    //Get the file name and check if it has a valid file extension
                    String name = f.getName();
                    //We only accept mp3, ogg and midi files in our case
                    if(name.endsWith(".mp3") || name.endsWith(".ogg") || name.endsWith(".midi")){
                        //If it's an audio file, create a SoundInformation object.
                        //Important: Since music tracks are usually bigger sound files, we set the "streamed" flag!
                        SoundInformation soundInfo = new SoundInformation(f);
                        soundInfo.setStreamed(true);
                        
                        //Add the SoundInformation to an ArrayList (which contains all music tracks)
                        musicTracks.add(soundInfo);
                        
                        //Print debug info so we can see which tracks were loaded
                        System.out.println("Added track to jukebox: "+name);
                    }
                }
            }
            
            musicTrackNames = new String[musicTracks.size()];
            for(int i = 0; i < musicTracks.size(); i++){
                SoundInformation s = musicTracks.get(i);
                musicTrackNames[i] = "Track " + (i + 1) + ": " + s.getFilename();
            }
        }
        
        //Register EventListener (this class)
        registerEventListener(this);
    }
    
    /**
     * This method is called when the plugin gets unloaded. 
     * We don't need it, so we just keep it empty
     */
    @Override
    public void onDisable(){
        //...
    }
    
    /**
     * This event method is called when a player "interacts" with an object, i.e. 
     * when he pressed his interaction key (F) while looking at an object. We use this
     * event to check if the player interacts with a sign. If it is a sign, we check if
     * the sign contains the text "jukebox" - in that case, we treat the sign
     * as a music player.
     * @param event the PlayerObjectInteractionEvent, which provides all relevant information
     * about this event, e.g. which player interacted with an object, which object did
     * the player interact with (IDs, coordinates) etc.
     */
    @EventMethod(Threading.Sync)
    public void onObjectInteraction(PlayerObjectInteractionEvent event){
        //Get the player object, i.e. the player who interacted with the object
        Player player = event.getPlayer();
        
        //We want to check if the object is a sign. The easiest way to do that is to
        //get the object definition...
        Definitions.ObjectDefinition def = event.getObjectDefinition();
        
        //...and check if this object definition belongs to a sign
        if(def.isSign()){
            //Sign ID is stored in "object info", use it to get the actual "Sign"
            Sign sign = getWorld().getSign(event.getObjectInfoID());
            
            //Always a good idea to check for null...
            if(sign != null){
                //Get the text of the sign
                String text = sign.getText();
                
                //Check if text is not null and if it contains the string "jukebox".
                //Since we don't care about upper/lower case, we just convert the
                //text to a lower case string
                if(text != null && text.toLowerCase().contains("jukebox")){
                    //Get the music player which is bound to this sign ID
                    MusicPlayer musicPlayer = musicPlayers.get(sign.getID());
                    
                    //If no music player exists, we create a new one and store it in our hash map
                    if(musicPlayer == null){
                        musicPlayer = new MusicPlayer(this, sign.getID(), event.getObjectPosition());
                        musicPlayers.put(sign.getID(), musicPlayer);
                    }
                    
                    //Check if music player is playing: in this case, stop the music player
                    if(musicPlayer.isPlaying()){
                        musicPlayer.stop();
                        
                        //Show a status message for the player who stopped the music player
                        player.showStatusMessage("[#FFAA00]Music player turned off", 2);
                    }
                    //Otherwise start the music player (more precisely, show a context menu containing all music tracks to the player)
                    else{
                        //Since we're referencing our musicPlayer variable from a lambda expression,
                        //it has to be final - so we just create a new, final variable for that
                        final MusicPlayer mp = musicPlayer;
                        
                        //Show a context menu for the player, so he can select a song there
                        player.showContextMenu(musicTrackNames, (String selection) -> {
                            //Since we just get a string (the string the player selected), 
                            //we have to go through all songs and check which song was actually selected.
                            //But first check if selection is not null (i.e. player just closed the menu)
                            if(selection != null && !selection.isEmpty()){
                                int track = -1;
                                for(int i = 0; i < musicTrackNames.length; i++){
                                    if(musicTrackNames[i].equals(selection)){
                                        //Track is found, so store it in our "track" variable and cancel/break the loop here
                                        track = i;
                                        break;
                                    }
                                }
                                
                                //Before starting the track, check again if the music player is still not playing.
                                //It's possible that another player selected a track at the same time while our track
                                //selection was open. In order to avoid two tracks to play simultaneously,
                                //we check again if the music player is playing.
                                if(mp.isPlaying()) return;
                                
                                //Play the track that was selected by the player
                                if(track >= 0){
                                    mp.play(track);
                                    
                                    //Show a status message for the player who started the track
                                    player.showStatusMessage("[#FFAA00]Now playing:\n[#33AABF]\"" + musicTrackNames[track] + "\"", 4);
                                }
                            }
                        });
                    }
                    
                    //Cancel the event, so the player cannot change the sign text
                    event.setCancelled(true);
                }
            }
        }
    }
    
    /**
     * This event method is called when a player joins the game and the loading
     * process is completed (i.e. once the player leaves the loading screen). This event
     * is only called once when the player connects to the server (or loads the world). 
     * If the player dies and respawns, the PlayerRespawnEvent is called instead.
     * 
     * We use this event to check if there are currently any music tracks playing, and if so,
     * we play the music track for the new player as well.
     * 
     * @param event the PlayerSpawnEvent which gives access to the player who just spawned
     */
    @EventMethod(Threading.Sync)
    public void onPlayerSpawn(PlayerSpawnEvent event){
        //Get the player who just spawned
        Player player = event.getPlayer();
        
        //Check if there is currently a music player playing (we iterate over our
        //"musicPlayers" hash map and check if one of it has the "isPlaying" variable set
        for(MusicPlayer musicPlayer : musicPlayers.values()){
            if(musicPlayer.isPlaying()){
                //Get the SoundInformation of the music track which is currently playing
                SoundInformation musicTrack = musicTracks.get(musicPlayer.getCurrentTrack());
                
                //Get the name of the attribute where we store the sound ID
                String attribute = musicPlayer.getAttributeName();
                
                //Check if the player already has the attribute set for some reason
                if(player.hasAttribute(attribute)){
                    //Attribute is already in use... so we get the soundID from the attribute...
                    int soundID = (int) player.getAttribute(attribute);
                    
                    //...and stop the sound for the player
                    player.stopSound(soundID);
                }
                
                //We start playing the music track for the player
                int soundID = player.playSound(musicTrack, true, MusicPlayer.DEFAULT_VOLUME, 1f, MusicPlayer.MIN_RANGE, MusicPlayer.MAX_RANGE, musicPlayer.getPosition());
                
                //Since we want to keep the music track in sync with other players, we can't just
                //start the track from the beginning. Instead we have to find out how many seconds
                //have passed since the music player was started. We have the "playStartTime" stored
                //(milliseconds), so we can just grab the current time millis stamp (Java System method),
                //subtract the playStartTime and multiply it with 0.001f (to convert the milliseconds to seconds)
                float timeline = (System.currentTimeMillis() - musicPlayer.getPlayStartTime()) * 0.001f;
                
                //Set the timeline position. Not a relative value
                player.setSoundTimelinePosition(soundID, timeline, false);
                
                //Finally store the soundID as attribute for the player
                player.setAttribute(attribute, soundID);
            }
        }
    }
    
    /**
     * This event method is called when an object gets destroyed. First we're going
     * to check if the destroyed object is a sign. In that case, we call our universal
     * "removeMusicPlayer()" method which checks if there is actually a music player
     * assigned to that object (and removes it accordingly)
     * @param event the PlayerDestroyObjectEvent, which provides access to all relevant
     * information about this event (e.g. which object was destroyed, IDs, coordinate, 
     * which player destroyed this object etc).
     */
    @EventMethod(Threading.Sync)
    public void onObjectDestroy(PlayerDestroyObjectEvent event){
        //Check if object is a sign - in that case, we check if it's a music player and remove it accordingly.
        //To find out if the destroyed object is a sign, we can use the object definition.
        if(event.getObjectDefinition().isSign()){
            removeMusicPlayer(event.getObjectInfoID());
        }
    }
    
    /**
     * This event method is called when an object gets removed (e.g. picked up). First we're going
     * to check if the removed object is a sign. In that case, we call our universal
     * "removeMusicPlayer()" method which checks if there is actually a music player
     * assigned to that object (and removes it accordingly)
     * @param event the PlayerRemoveObjectEvent, which provides access to all relevant
     * information about this event (e.g. which object was removed, IDs, coordinate, 
     * which player removed this object etc).
     */
    @EventMethod(Threading.Sync)
    public void onObjectRemove(PlayerRemoveObjectEvent event){
        //Check if object is a sign - in that case, we check if it's a music player and remove it accordingly
        if(event.getObjectDefinition().isSign()){
            removeMusicPlayer(event.getObjectInfoID());
        }
    }
    
    /**
     * Removes a music player, or more precisely, checks if there is a music player
     * assigned to that sign ID. In that case, any currently playing tracks will be stopped
     * and the music player will be deleted.
     * @param id the ID we want to check.
     */
    private void removeMusicPlayer(int id){
        //Look if a music player is assigned to that ID
        MusicPlayer musicPlayer = musicPlayers.get(id);
        if(musicPlayer != null){
            //Check if music player is currently playing
            if(musicPlayer.isPlaying()){
                //In that case, turn it off
                musicPlayer.stop();
            }
            
            //Remove the music player from our "musicPlayers" hash map
            musicPlayers.remove(id);
        }
    }
    
    /**
     * Gets the path to the music folder. The path is either stored in a "Music.txt"
     * file in the plugin folder (which just contains the path, e.g. "C:/Music/"), or
     * alternatively (if the file does not exist) the game uses the "Music" subfolder in the plugin folder.
     * @return a file object which represents the music folder, or null if no music folder exists.
     */
    private File getMusicFolder(){
        //We store the path to the music in a file called "path.txt"
        File musicPath = new File(getPath() + "/Music.txt");
        
        //Check if that file exists
        if(musicPath.exists()){
            //Read the content of the file (which is supposed to be the target file path)
            String path = Utils.FileUtils.readStringFromFile(musicPath);
            
            //Check if the path is not null and not empty
            if(path != null && !path.isEmpty()){
                File musicDirectory = new File(path);
                
                //Check if the path does actually exists and if it's a directory
                if(musicDirectory.exists() && musicDirectory.isDirectory()){
                    return musicDirectory;
                }
            }
        }
        //If the file does not exist, we look for a "Music" subfolder instead
        else{
            File musicDirectory = new File(getPath() + "/Music/");
            
            //If the "Music" subfolder exists, we return it
            if(musicDirectory.exists()){
                return musicDirectory;
            }
        }
        
        //Return null if music path does not exist
        return null;
    }
    
    
}
