package net.risingworld.api.example.custommusic;

import java.io.File;
import java.util.ArrayList;
import net.risingworld.api.Plugin;
import net.risingworld.api.Timer;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.player.PlayerSpawnEvent;
import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.SoundInformation;
import net.risingworld.api.utils.Utils;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * 
 * @author red51
 */

public class CustomGameMusic extends Plugin implements Listener{
    
    //An ArrayList containing all music tracks which were found in our music folder
    public ArrayList<SoundInformation> musicTracks = new ArrayList<>();
    
    //We use a final variable to store the name of our "current track attribute"
    //(so it's easier to use or change it)
    private static final String CURRENTTRACK_ATTRIBUTE = "custom_gamemusic_currenttrack";
    
    /**
     * This method is called when the plugin is loaded / initialized. We use it to search all
     * music tracks that can be used by our plugin. In addition to that, we want our plugin to
     * listen for "events", at least we need to know when a player spawns, so we register
     * our plugin as "EventListener" at the bottom of this method
     */
    @Override
    public void onEnable(){
        //Get the music folder
        File musicFolder = getMusicFolder();
        
        //If the directory exists, we will read all files from it
        if(musicFolder != null && musicFolder.exists() && musicFolder.isDirectory()){
            File[] files = musicFolder.listFiles();
            for(File f : files){
                //We will ignore directories inside our music folder
                if(!f.isDirectory()){
                    //Get the file name and check if it has a valid file extension
                    String name = f.getName();
                    //We only accept mp3, ogg and midi files in our case
                    if(name.endsWith(".mp3") || name.endsWith(".ogg") || name.endsWith(".midi")){
                        //If it's an audio file, create a SoundInformation object.
                        //Important: Since music tracks are usually bigger sound files, we set the "streamed" flag!
                        SoundInformation soundInfo = new SoundInformation(f);
                        soundInfo.setStreamed(true);
                        
                        //Add the SoundInformation to an ArrayList (which contains all music tracks)
                        musicTracks.add(soundInfo);
                        
                        //Print debug info so we can see which tracks were loaded
                        System.out.println("Loaded custom music track: "+name);
                    }
                }
            }
        }
        
        //Register EventListener (this class)
        registerEventListener(this);
    }
    
    /**
     * This method is called when the plugin gets unloaded. 
     * We don't need it, so we just keep it empty
     */
    @Override
    public void onDisable(){
        //...
    }
    
    /**
     * This event method is called when a player joins the game and the loading
     * process is completed (i.e. once the player leaves the loading screen). This event
     * is only called once when the player connects to the server (or loads the world). 
     * If the player dies and respawns, the PlayerRespawnEvent is called instead.
     * 
     * We use this event to disable the default game music for the player and to start a timer
     * which handles the new custom music tracks
     * 
     * @param event the PlayerSpawnEvent which gives access to the player who just spawned
     */
    @EventMethod
    public void onPlayerSpawn(PlayerSpawnEvent event){
        //Get the player object (the player who spawned)
        Player player = event.getPlayer();
        
        //First of all, disable the default ingame music so it does not overlap with our custom music
        player.disableGameMusic();
        
        //We check if there are any custom music tracks available at all
        if(!musicTracks.isEmpty()){
            //Now we want to select a random first track and store it as player attribute (we use a unique name)
            player.setAttribute(CURRENTTRACK_ATTRIBUTE, Utils.MathUtils.nextRandomInt(0, musicTracks.size() - 1));

            //Now start a timer: in our example, it will play a new track every 10 minutes (600 seconds). 
            //Initial delay is 30 seconds (i.e. the first track already starts after 20 seconds)
            Timer timer = new Timer(600, 20, -1, null);

            //Set the task separately, so we can access the timer object and kill the timer if the player is no longer connected
            timer.setTask(() -> {
                //Check if player is offline, in that case, kill this timer (to save resources) and return
                if(!player.isConnected()){
                    timer.kill();
                    return;
                }
                
                //Get the current track (int stored as attribute, see above)
                int currentTrack = (int) player.getAttribute(CURRENTTRACK_ATTRIBUTE);

                //We don't want to play a random track, since this could cause the same
                //track to play twice in a row (or even more often). To do that, simply
                //increment the "currenttrack" attribute of the player, but check if
                //it's within the bounds of the musicTracks ArrayList (otherwise set to 0)
                if(++currentTrack >= musicTracks.size()) currentTrack = 0;
                
                //Remember to update the player attribute!
                player.setAttribute(CURRENTTRACK_ATTRIBUTE, currentTrack);
                
                //Get the music track (i.e. SoundInformation which is stored in our ArrayList)
                SoundInformation track = musicTracks.get(currentTrack);
                
                //We want to know the audio volume setting the client has set for music.
                //So we retrieve his "audio_music_volume" setting from the "config.properties"
                //file and use the value as our music volume. Of course we could just use
                //a fixed volume value, but this seems to be more appropriate
                player.getOption("audio_music_volume", (value) -> {
                    //Check if the value is numeric
                    if(value != null && Utils.StringUtils.isNumeric(value)){
                        //Parse volume value to float
                        float volume = Float.parseFloat(value);
                        
                        //Warning: The audio settings are stored as a value between 0-100,
                        //but our function requires a value between 0-1, so we have to
                        //multiply it with 0.01f
                        volume *= 0.01f;
                        
                        //Play the music track (set loop to false) as a 2D sound (i.e. position = null)
                        player.playSound(track, false, volume, 1f, 0f, 0f, null);
                        
                        //Optionally you could send a message to the player and tell him which track is currently playing
                        //player.sendTextMessage("\n[#42f4d4]MUSIC PLAYER - NOW PLAYING: \n[#42f4d4]\"" + track.getFilename() + "\"");
                    }
                });
            });
            
            //As always, remember to start the timer ;)
            timer.start();
        }
    }
    
    
    /**
     * Gets the path to the music folder. The path is either stored in a "Music.txt"
     * file in the plugin folder (which just contains the path, e.g. "C:/Music/"), or
     * alternatively (if the file does not exist) the game uses the "Music" subfolder in the plugin folder.
     * @return a file object which represents the music folder, or null if no music folder exists.
     */
    private File getMusicFolder(){
        //We store the path to the music in a file called "path.txt"
        File musicPath = new File(getPath() + "/Music.txt");
        
        //Check if that file exists
        if(musicPath.exists()){
            //Read the content of the file (which is supposed to be the target file path)
            String path = Utils.FileUtils.readStringFromFile(musicPath);
            
            //Check if the path is not null and not empty
            if(path != null && !path.isEmpty()){
                File musicDirectory = new File(path);
                
                //Check if the path does actually exists and if it's a directory
                if(musicDirectory.exists() && musicDirectory.isDirectory()){
                    return musicDirectory;
                }
            }
        }
        //If the file does not exist, we look for a "Music" subfolder instead
        else{
            File musicDirectory = new File(getPath() + "/Music/");
            
            //If the "Music" subfolder exists, we return it
            if(musicDirectory.exists()){
                return musicDirectory;
            }
        }
        
        //Return null if music path does not exist
        return null;
    }
    
}
