package net.risingworld.api.example.position;

import net.risingworld.api.Plugin;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.player.PlayerChangePositionEvent;
import net.risingworld.api.events.player.PlayerSpawnEvent;
import net.risingworld.api.gui.Font;
import net.risingworld.api.gui.GuiLabel;
import net.risingworld.api.gui.GuiPanel;
import net.risingworld.api.gui.PivotPosition;
import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.Vector3f;

/**
 * @author red51
 */

public class Position extends Plugin implements Listener{

    @Override
    public void onEnable(){
        //We just register our plugin class as event listener here
        registerEventListener(this);
    }

    @Override
    public void onDisable(){
        //...
    }
    
    /**
     * This event method is called when a player joins the game and the loading
     * process is completed (i.e. once the player leaves the loading screen). This event
     * is only called once when the player connects to the server (or loads the world). 
     * If the player dies and respawns, the PlayerRespawnEvent is called instead.
     * 
     * We use this event to create the required gui elements for the player (we only create
     * them once and update them then accordingly).
     * 
     * @param event the PlayerSpawnEvent which gives access to the player who just spawned
     */
    @EventMethod
    public void onPlayerSpawn(PlayerSpawnEvent event){
        //Get the player object (the player who spawned)
        Player player = event.getPlayer();
        
        //First we create a GUI panel (which serves as a background panel)
        GuiPanel panel = new GuiPanel(0.5f, 0.04f, true, 300, 30, false);
        
        //We want the panel to be centered (i.e. using "center" pivot)
        panel.setPivot(PivotPosition.Center);
        
        //We set a black color (rgb 0 0 0) and very transparent alpha value (35% opacity)
        panel.setColor(0f, 0f, 0f, 0.35f);
        
        
        //Now we create the actual label which shows our coordinates and add it to the panel
        GuiLabel posLabel = new GuiLabel(0.5f, 0.5f, true);
        
        //Centered pivot
        posLabel.setPivot(PivotPosition.Center);
        
        //We set a bold mono font
        posLabel.setFont(Font.DefaultMono_Bold);
        
        //A font size of 20 seems to be reasonable
        posLabel.setFontSize(20);
        
        //We set a white and fully opaque font color 
        posLabel.setFontColor(1f, 1f, 1f, 1f);
        
        //Add the label as a child to the panel
        panel.addChild(posLabel);
        
        
        //Now add the panel to the player screen
        player.addGuiElement(panel);
        
        //Even though the label is a child of the panel, we still have to add it manually
        player.addGuiElement(posLabel);
        
        //We store the label object as an attribute, so we can access it later
        player.setAttribute("net.rw.api.pos.label", posLabel);
    }
    
    /**
     * This event method is called when the player changes his position. 
     * It's an event which is frequently called, so it's not recommendable to put
     * any "heavy" or "expensive" code there.
     * @param event the PlayerChangePositionEvent which gives access to the player who 
     * changed the position.
     */
    @EventMethod
    public void onPlayerChangePosition(PlayerChangePositionEvent event){
        //Get the player in question
        Player player = event.getPlayer();
        
        //Get the attribute which was stored previously (in the onPlayerSpawn() method) which
        //is supposed to be our GuiLabel. We don't create a new label every time since that
        //would waste a lot of performance.
        //Since an attribute is always stored as a universal "Object", we have to be careful
        //with casting. The best approach is to retrieve the universal "Object" first
        Object attribute = player.getAttribute("net.rw.api.pos.label");
        
        //Now check if the attribute is not null and if it's an instance of the "GuiLabel" class
        //(i.e. if it is actually a label). There is always a chance that another plugin
        //used the same attribute name accidentally and stored a different data type there, 
        //and since we want to avoid any exceptions, we are very careful about the attribute type
        if(attribute != null && attribute instanceof GuiLabel){
            //Now we can safely cast the universal object to a "GuiLabel"
            GuiLabel label = (GuiLabel) attribute;
            
            //Get the current player position (returned as a Vector3f)
            Vector3f pos = player.getPosition();
            
            //Our position vector holds the X, Y and Z position as float values.
            //We want to avoid that our position label shows long numbers (e.g. 0.53242342),
            //so we use the "String.format()" function which displays a string in a
            //certain format (in this case, we tell it to show the float values with
            //a single decimal place only)
            label.setText(String.format("%.1f   %.1f   %.1f", pos.x, pos.y, pos.z));
        }
    }
    
}
