package net.risingworld.api.example.discord;

import java.io.File;
import net.risingworld.api.Plugin;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.player.PlayerCommandEvent;
import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.Utils;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * Since this is just a small plugin, we don't want to create a separate event
 * listener, so we just declared this class as our event listener (by adding
 * "implements Listener", i.e. this plugin implements the Listener interface).
 * 
 * @author red51
 */

public class ConnectDiscord extends Plugin implements Listener{

    @Override
    public void onEnable(){
        //Check if "config.txt" file exists, otherwise print a warning message and return
        File configFile = new File(getPath() + "/config.txt");
        if(!configFile.exists()){
            System.err.println("WARNING: ConnectToDiscord Plugin - no \"config.txt\" file found in plugin folder!");
            return;
        }
        
        //Our plugin main class is also our event listener (it implements the
        //"Listener" interface), so we have to register it accordingly:
        registerEventListener(this);
    }

    @Override
    public void onDisable(){
        //Nothing we want to do when the plugin gets unloaded, so we just 
        //keep this method blank...
    }
    
    /**
     * This is our "CommandEvent" method. It is called when a player types
     * a command into chat. The provided "event" object gives access to the various
     * event information (command, player). 
     * Note that you have to add the annotation "@EventMethod" to all event methods,
     * otherwise they will not be recognized (and as a result, not be called when
     * the particular event gets triggered).
     * @param event our event object, which provides access to the event information
     */
    @EventMethod
    public void onPlayerCommand(PlayerCommandEvent event){
        //This string contains the full command the player has entered
        String command = event.getCommand();
        
        //Get the player who entered the command
        Player player = event.getPlayer();
        
        //Player has to type "/discord", then he will be connected to discord (via browser)
        if(command.equals("/discord")){
            //We want to get the discord url from a config file in the plugin folder
            File configFile = new File(getPath() + "/config.txt");
            
            //There is a convenient function in the Utils class which allows you
            //to get text content from a file
            String url = Utils.FileUtils.readStringFromFile(configFile);
            
            //Check if url is not null (i.e. file did not exist)
            if(url != null){
                //Remove all whitespaces (optional)
                url = Utils.StringUtils.removeAllWhitespaces(url);
                
                //There is a convenient method to connect the client to discord, just provide
                //the url of the discord server.
                player.connectToDiscord(url);
            }
        }
    }

}
