package net.risingworld.api.example.ts;

import java.io.File;
import net.risingworld.api.Plugin;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.player.PlayerCommandEvent;
import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.Utils;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).
 * Since this is just a small plugin, we don't want to create a separate event
 * listener, so we just declared this class as our event listener (by adding
 * "implements Listener", i.e. this plugin implements the Listener interface).
 * 
 * @author red51
 */

public class ConnectTS extends Plugin implements Listener{

    @Override
    public void onEnable(){
        //Check if "config.txt" file exists, otherwise print a warning message and return
        File configFile = new File(getPath() + "/config.txt");
        if(!configFile.exists()){
            System.err.println("WARNING: ConnectToTeamSpeak Plugin - no \"config.txt\" file found in plugin folder!");
            return;
        }
        
        //Our plugin main class is also our event listener (it implements the
        //"Listener" interface), so we have to register it accordingly:
        registerEventListener(this);
    }

    @Override
    public void onDisable(){
        //Nothing we want to do when the plugin gets unloaded, so we just 
        //keep this method blank...
    }
    
    /**
     * This is our "CommandEvent" method. It is called once the player types
     * a command into chat. The provided "event" object gives access to the various
     * event information (entered command, player). 
     * Note that you have to add the annotation "@EventMethod" to all event methods,
     * otherwise they will not be recognized (and as a result, not be called when
     * the particular event gets triggered).
     * @param event our event object, which provides access to the event information
     */
    @EventMethod
    public void onPlayerCommand(PlayerCommandEvent event){
        //This string contains the full command the player has entered
        String command = event.getCommand();
        
        //Get the player who entered the command
        Player player = event.getPlayer();
        
        //Player has to type "/ts" or "/teamspeak", then he will be connected to a teamspeak server
        if(command.equals("/ts") || command.equals("/teamspeak")){
            //We want to get the TeamSpeak IP, port and channel from a text file (called config.txt)
            File configFile = new File(getPath() + "/config.txt");
            
            //There is a convenient function in the Utils class which allows you
            //to get text content from a file
            String content = Utils.FileUtils.readStringFromFile(configFile);
            
            //Check if content is not null (i.e. file did not exist)
            if(content != null){
                String ip = null;
                String port = null;
                String channel = null;
                
                //Get IP, port and channel (optional) from the text file
                String[] lines = Utils.StringUtils.getLines(content);
                for(String line : lines){
                    //Convert the line to lower case, so it doesn't matter if the user wrote "IP" or "ip"
                    String lline = line.toLowerCase();
                    if(lline.toLowerCase().startsWith("ip:")){
                        ip = Utils.StringUtils.removeAllWhitespaces(line.substring(3));
                    }
                    else if(lline.startsWith("port:")){
                        port = Utils.StringUtils.removeAllWhitespaces(line.substring(5));
                    }
                    else if(lline.startsWith("channel:")){
                        //Since the channel can contain white spaces, we only remove the leading white spaces
                        channel = Utils.StringUtils.removeAllLeadingWhitespaces(line.substring(8));
                    }
                }
                
                //Check if at least IP and port are not null, and check if port is an integer
                if(ip != null && port != null && Utils.StringUtils.isInteger(port)){
                    //There is a convenient method to connect the client to a teamspeak
                    //server. You just have to provide the IP of the TeamSpeak server, the
                    //port (cast to int), and optionally the target channel (the player will be moved to
                    //this channel automatically).
                    //Note: This does only work if the client uses Windows
                    player.connectToTeamSpeak(ip, Integer.parseInt(port), channel);
                }
            }
        }
    }

}
