/*
 * TimeSpeedChanger.java
 *
 * User: Carsten
 * Date: 26.04.2017
 * Version: 1.2.0
 */

package carsten.risingworld.tsc;

import net.risingworld.api.Plugin;
import net.risingworld.api.Server;
import net.risingworld.api.Timer;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.Threading;
import net.risingworld.api.events.player.PlayerCommandEvent;
import net.risingworld.api.events.player.PlayerSleepEvent;
import net.risingworld.api.events.player.PlayerSpawnEvent;
import net.risingworld.api.gui.Font;
import net.risingworld.api.gui.GuiLabel;
import net.risingworld.api.gui.PivotPosition;
import net.risingworld.api.objects.Player;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.FileSystems;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

public class TimeSpeedChanger extends Plugin implements Listener
{
  private static final String        SEP      = FileSystems.getDefault().getSeparator();
  private static final String        FILENAME = "tsc.prefs";
  private static final DecimalFormat dig2     = new DecimalFormat("00");

  private static final String MODE_GT = "gametime";
  private static final String MODE_RT = "realtime";
  private static final String MODE_DN = "daynight";
  private static final List<String> MODE_LIST;

  static
  {
    MODE_LIST = new ArrayList<>(3);
    MODE_LIST.add(MODE_GT);
    MODE_LIST.add(MODE_RT);
    MODE_LIST.add(MODE_DN);
  }

  private static final String CMD_HELP  = "command_help";
  private static final String CMD_CLCK  = "command_clock";
  private static final String CMD_GAME  = "command_gametime";
  private static final String CMD_REAL  = "command_realtime";
  private static final String CMD_DAYN  = "command_daynight";
  private static final String MODE      = "mode";
  private static final String TSP_DAY   = "timespeed_day";
  private static final String TSP_NIGHT = "timespeed_night";
  private static final String CLCK_VIS  = "clock_visible";
  private static final String CLCK_X    = "clock_posx";
  private static final String CLCK_Y    = "clock_posy";
  private static final String CLCK_SIZE = "clock_size";

  private final Runnable   timerTask_FP;
  private       Properties props;

  private float origTimeSpeed;

  private int second;
  private int minute;
  private int hour;

  public TimeSpeedChanger()
  {
    super();

    timerTask_FP = this::timerTask;
  }

  @Override
  public void onEnable()
  {
    Server server = getServer();

    // rescue the timespeed defined in Rising World's config.properties
    origTimeSpeed = server.getGameTimeSpeed();

    props = new Properties();

    loadSettings();

    registerEventListener(this);
  }

  @Override
  public void onDisable()
  {

  }

  @EventMethod(Threading.Sync)
  public void onPlayerSpawn(PlayerSpawnEvent evt)
  {
    Player player = evt.getPlayer();
    // V1.1: only admin
    if(!player.isAdmin())
    {
      return;
    }

    GuiLabel label = new GuiLabel(getFloat(props.getProperty(CLCK_X)), getFloat(props.getProperty(CLCK_Y)), true);
    label.setPivot(PivotPosition.TopLeft);
    label.setFont(Font.DefaultMono_Bold);
    label.setFontSize(getInteger(props.getProperty(CLCK_SIZE)));
    label.setClickable(false);
    label.setBorderColor(0.2f, 0.2f, 0.2f, 0.5f);
    label.setBorderThickness(0.6f, true);
    label.setText("          ");

    player.addGuiElement(label);
    player.setAttribute("tsc_clock_label", label);

    setGuiClockVisible(player);

    Timer timer = new Timer(1.0f, 0, -1, timerTask_FP);
    timer.start();

    updateRealTime();
  }

  @EventMethod(Threading.Async)
  public void onPlayerCommand(PlayerCommandEvent evt)
  {
    Player player = evt.getPlayer();
    if(!player.isAdmin())
    {
      return;
    }

    StringTokenizer st = new StringTokenizer(evt.getCommand(), " /");
    if(st.countTokens() <= 0)
    {
      return;
    }

    String cmd = st.nextToken();

    if(cmd.equals(props.getProperty(CMD_HELP)))
    {
      printHelp(player);
    }
    else if(cmd.equals(props.getProperty(CMD_CLCK)))
    {
      toggleClock(player);
    }
    else if(cmd.equals(props.getProperty(CMD_GAME)))
    {
      setGameTime();
    }
    else if(cmd.equals(props.getProperty(CMD_REAL)))
    {
      setRealTime();
    }
    else if(cmd.equals(props.getProperty(CMD_DAYN)))
    {
      if(st.countTokens() != 2)
      {
        printMode(player);
        return;
      }
      float day;
      float night;

      try
      {
        day = Float.parseFloat(st.nextToken());
        night = Float.parseFloat(st.nextToken());
      }
      catch(NumberFormatException nfex)
      {
        player.sendTextMessage("error: not a number");
        return;
      }

      setDayNight(day, night);
    }

    saveSettings();
  }

  private void printMode(final Player player) {
    String txt = "[#ffff0f]TimeSpeedChanger V1.1: mode=" + props.getProperty(MODE);
    if(props.getProperty(MODE).equals(MODE_DN))
    {
      txt += " speed day: " + props.getProperty(TSP_DAY) + " speed night: " + props.getProperty(TSP_NIGHT);
    }
    player.sendTextMessage(txt);
    player.sendTextMessage("");
  }

  @EventMethod(Threading.Async)
  public void onPlayerSleep(PlayerSleepEvent evt)
  {
    if(!evt.getPlayer().isAdmin())
    {
      return;
    }
    if(evt.isSleeping())
    {
      return;
    }

    updateRealTime();
  }

  private void timerTask()
  {
    updateTime();

    updateLabelAllPlayers();

    updateDayNight();
  }

  private void updateTime()
  {
    Calendar calendar = Calendar.getInstance();

    second = calendar.get(Calendar.SECOND);
    minute = calendar.get(Calendar.MINUTE);
    hour = calendar.get(Calendar.HOUR_OF_DAY);
  }

  private void updateLabelAllPlayers()
  {
    if(!isClockVisible())
    {
      return;
    }

    for(Player player : getServer().getAllPlayers())
    {
      if(player.isAdmin())
      {
        updateLabel(player);
        break;
      }
    }
  }

  private void updateLabel(final Player player)
  {
    GuiLabel label = (GuiLabel) player.getAttribute("tsc_clock_label");
    if(label == null)
    {
      return;
    }

    String time = " " + dig2.format(hour) + ":" + dig2.format(minute) + ":" + dig2.format(second) + " ";
    label.setText(time);
  }

  private void updateRealTime()
  {
    if(!MODE_RT.equals(props.getProperty(MODE)))
    {
      return;
    }

    updateTime();

    setRealTime();
  }

  private void updateDayNight()
  {
    if(!MODE_DN.equals(props.getProperty(MODE)))
    {
      return;
    }

    Server server = getServer();

    int hours = server.getGameTime().getHours();
    if(hours >= 8 && hours < 22)
    {
      server.setGameTimeSpeed(Float.parseFloat(props.getProperty(TSP_DAY)));
    }
    else
    {
      server.setGameTimeSpeed(Float.parseFloat(props.getProperty(TSP_NIGHT)));
    }
  }

  private boolean isClockVisible()
  {
    return Boolean.parseBoolean(props.getProperty(CLCK_VIS));
  }

  private void toggleClock(final Player player)
  {
    props.setProperty(CLCK_VIS, Boolean.toString(!isClockVisible()));
    setGuiClockVisible(player);
  }

  private void setGuiClockVisible(final Player player)
  {
    GuiLabel label = (GuiLabel) player.getAttribute("tsc_clock_label");

    if(label == null)
    {
      return;
    }
    player.setGuiElementVisible(label, Boolean.parseBoolean(props.getProperty(CLCK_VIS)));
  }

  private void setGameTime()
  {
    props.setProperty(MODE, MODE_GT);
    getServer().setGameTimeSpeed(origTimeSpeed);
  }

  private void setRealTime()
  {
    props.setProperty(MODE, MODE_RT);
    Server server = getServer();
    server.setGameTimeSpeed(60.0f);
    server.setGameTime(hour, minute);
  }

  private void setDayNight(final float p_day, final float p_night)
  {
    if(p_day <= 0 || p_night <= 0)
    {
      return;
    }
    props.setProperty(MODE, MODE_DN);
    props.setProperty(TSP_DAY, Float.toString(p_day));
    props.setProperty(TSP_NIGHT, Float.toString(p_night));
  }

  private void printHelp(final Player player)
  {
    player.sendTextMessage("");
    player.sendTextMessage("");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("*                TimeSpeedChanger V1.1 Plugin - help page               *");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("* ");
    player.sendTextMessage("* Available commands:");
    player.sendTextMessage("* /" + props.getProperty(CMD_HELP));
    player.sendTextMessage("*   show this help page");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_CLCK));
    player.sendTextMessage("*   show/ hide realtime clock on screen");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_GAME));
    player.sendTextMessage("*   switch to normal ingame timespeed");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_REAL));
    player.sendTextMessage("*   switch to current time and realtime timespeed");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_DAYN) + " x.xx y.yy");
    player.sendTextMessage("*   set distinct timespeed for day (08:00-22:00) and night (22:00-08:00),");
    player.sendTextMessage("*   the amount of realtime seconds for one ingame minute, f.e.:");
    player.sendTextMessage("*   /" + props.getProperty(CMD_DAYN) + " 3.0 2.0");
    player.sendTextMessage("*   a minute at daytime takes 3 seconds, at night it takes 2 seconds");
    player.sendTextMessage("* ");
    player.sendTextMessage("* The readme.txt in the plugin folder contains further information.");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("");
    player.sendTextMessage("(Activate chat and press <Page Up> <Page Down> to scroll) ");
    player.sendTextMessage("");
  }

  private void loadSettings()
  {
    loadPreferences(FILENAME, props);
    checkSettings();
    // save all properties to amend corrupt or missing ones
    savePreferences(FILENAME, props);
  }

  private void saveSettings()
  {
    savePreferences(FILENAME, props);
  }

  private void checkSettings()
  {
    checkStringValue(props, CMD_HELP, "tschelp");
    checkStringValue(props, CMD_CLCK, "tscclock");
    checkStringValue(props, CMD_GAME, "tscgametime");
    checkStringValue(props, CMD_REAL, "tscrealtime");
    checkStringValue(props, CMD_DAYN, "tscdaynight");

    String str = props.getProperty(MODE);
    if(str == null || !MODE_LIST.contains(str))
    {
      props.setProperty(MODE, MODE_GT);
    }

    str = props.getProperty(CLCK_VIS);
    if(str == null || !(str.equals("true") || str.equals("false")))
    {
      props.setProperty(CLCK_VIS, "true");
    }

    checkFloatRange(props, TSP_DAY, 0.01f, 999.99f, 1.75f);
    checkFloatRange(props, TSP_NIGHT, 0.01f, 999.99f, 1.75f);
    checkFloatRange(props, CLCK_X, 0.0f, 1.0f, 0.0f);
    checkFloatRange(props, CLCK_Y, 0.0f, 1.0f, 1.0f);

    checkIntegerRange(props, CLCK_SIZE, 1, 100, 20);
  }

  private void checkStringValue(final Properties props, final String key, final String defaultValue)
  {
    String str = props.getProperty(key);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(key, defaultValue);
    }
  }

  private void checkFloatRange(final Properties props, final String key, final float min, final float max, final float def)
  {
    float f = getFloat(props.getProperty(key));
    if(f < min || f > max)
    {
      props.setProperty(key, Float.toString(def));
    }
  }

  private void checkIntegerRange(final Properties props, final String key, final int min, final int max, final int def)
  {
    int i = getInteger(props.getProperty(key));
    if(i < min || i > max)
    {
      props.setProperty(key, Integer.toString(def));
    }
  }

  private static float getFloat(String str)
  {
    if(str == null)
    {
      str = "";
    }
    float f;
    try
    {
      f = Float.parseFloat(str);
    }
    catch(NumberFormatException ex)
    {
      f = Float.POSITIVE_INFINITY;
    }

    return f;
  }

  private static int getInteger(String str)
  {
    if(str == null)
    {
      str = "";
    }
    int i;
    try
    {
      i = Integer.parseInt(str);
    }
    catch(NumberFormatException ex)
    {
      i = 0;
    }

    return i;
  }

  private void loadPreferences(final String filename, final Properties props)
  {
    String path = getPath();
    if(path == null || path.length() < 1)
    {
      return;
    }
    String filepath = path + SEP + filename;
    File   file     = new File(filepath);
    if(!file.exists())
    {
      savePreferences(filename, props);
    }
    try(Reader reader = new FileReader(file))
    {
      props.load(reader);
    }
    catch(IOException ex)
    {
      // no-op
    }
  }

  private void savePreferences(final String filename, final Properties props)
  {
    String path = getPath();
    if(path == null || path.length() < 1)
    {
      return;
    }
    String filepath = path + SEP + filename;

    File file = new File(filepath);
    try
    {
      //noinspection ResultOfMethodCallIgnored
      file.getParentFile().mkdirs();
      //noinspection ResultOfMethodCallIgnored
      file.createNewFile();
    }
    catch(IOException ex)
    {
      // no-op
    }
    try(Writer writer = new FileWriter(file, false))
    {
      props.store(writer, null);
      writer.flush();
      writer.close();
    }
    catch(IOException ex)
    {
      // no-op
    }
  }
}
