////////////////////////////////////////////////////////////////////////////////////
/////    Vehicle Shader for vehicles with different paintjobs                  /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Vertex-Shader                                                         /////
////////////////////////////////////////////////////////////////////////////////////

uniform mat4 g_WorldViewProjectionMatrix;
uniform mat4 g_WorldViewMatrix;
uniform mat3 g_WorldMatrixInverseTranspose;
uniform mat4 g_WorldMatrix;
uniform vec3 g_CameraPosition;

//Per-Vertex-Lighting (all static Lights + dynamic Pointlights)
#ifndef NUMPOINTLIGHTS
    #define NUMPOINTLIGHTS 1
#endif
uniform vec4 m_PointLightColor[NUMPOINTLIGHTS]; //RGB, A=LightRadius
uniform vec4 m_PointLightPosition[NUMPOINTLIGHTS]; //XYZ, W=InnerRadius

//Optional: Per-Vertex-Lighting (dynamic Spotlights)
#ifdef VERTEXLIGHTING
    #ifndef NUMSPOTLIGHTS
        #define NUMSPOTLIGHTS 1
    #endif
    uniform vec4 m_SpotLightColor[NUMSPOTLIGHTS]; //RGB, A=MaxLightRadius (frontal)
    uniform vec4 m_SpotLightPosition[NUMSPOTLIGHTS]; //XYZ, W=Range (raycasted)
    uniform vec4 m_SpotLightDirection[NUMSPOTLIGHTS]; //XYZ, W=OuterRadius
#endif

uniform vec4 m_Sunlight;
uniform vec4 m_Ambient;

uniform vec4 m_Fog;

#ifdef ENVIRONMENTMAPPING
    uniform vec3 m_Fresnel;
    varying vec4 envRef;
#endif

attribute vec3 inPosition;
attribute vec3 inNormal;
attribute vec4 inColor;
attribute vec2 inTexCoord;
attribute vec3 inTexCoord2;  //X=TextureLayer, Y=Transparency (only relevant for transparent parts), Z=ReflectionAmount (1=default, 0=off)

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 light;
varying vec2 texCoord;
varying vec3 texCoord2;
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const vec3 sunsetOrange = vec3(1.0, 0.78, 0.49);

void main(){
    vec4 pos = vec4(inPosition, 1.0);
    gl_Position = g_WorldViewProjectionMatrix * pos;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        positionProjectionSpace = gl_Position;
    #endif
    
    #ifdef FOG
        vec4 viewpos = g_WorldViewMatrix * pos;
        float dist = length(viewpos.xyz);
        fog = m_Fog;
        fog.w = exp2(-m_Fog.w * m_Fog.w * dist * dist * 1.442695);
        fog.w = clamp(fog.w, 0.0, 1.0);
    #endif
    
    position = inPosition;
    worldpos = vec3(g_WorldMatrix * pos);
    normal = normalize(g_WorldMatrixInverseTranspose * inNormal);
    texCoord = inTexCoord;
    texCoord2 = inTexCoord2;
    
    //SunlightColor calculation
    float sunY = m_Sunlight.y * -1.0;
    vec3 sunsetColor = mix(sunsetOrange, vec3(1.0,1.0,1.0), smoothstep(0.2, 0.3, sunY));
    vec3 sunLight = mix(vec3(0.05,0.25,0.68), sunsetColor, m_Sunlight.w);
    
    //StaticLight calculation
    light = vec3(inColor.w * 0.5 + (inColor.w * 0.5 * max(dot(m_Sunlight.xyz, normal) * -1.0 * m_Sunlight.w, 0.0)) + (m_Ambient.rgb * m_Ambient.a)) * sunLight;
    light += inColor.rgb;
    
    //Per-Vertex-Lighting calculation (dynamic Pointlights)
    for(int i = 0; i < NUMPOINTLIGHTS; i++){
        vec4 lightPos = m_PointLightPosition[i];
        vec4 lightDir = vec4((worldpos.xyz - lightPos.xyz), 0.0);
        float dist = distance(worldpos.xyz, lightPos.xyz);
        float intensity = (1.0 - smoothstep(lightPos.w, m_PointLightColor[i].w, dist)) * clamp((dot(normal, normalize(lightDir.xyz * -1.0)) + 0.25), 0.0, 1.0);
        light += (m_PointLightColor[i].xyz * intensity);
    }
    
    //If low-quality light is enabled: Per-Vertex-Lighting calculation (dynamic Spotlights)
    #ifdef VERTEXLIGHTING
        for(int i = 0; i < NUMSPOTLIGHTS; i++){
            if(m_SpotLightColor[i].w > 0.0){
                vec4 lightPos = m_SpotLightPosition[i];
                vec4 lightDir = m_SpotLightDirection[i];
                vec4 dir = vec4((worldpos.xyz - lightPos.xyz), 0.0);
                float dist = distance(worldpos.xyz, lightPos.xyz);
                float spotdot = dot(normalize(lightDir.xyz), normalize(dir.xyz));
                float intensity = smoothstep(1.0 - lightDir.w, 1.0, spotdot) * (1.0 - smoothstep(m_SpotLightColor[i].w * 0.5, m_SpotLightColor[i].w, dist)) * clamp((dot(normal, normalize(lightDir.xyz * -1.0)) + 0.25), 0.0, 1.0);
                intensity *= (1.0 - step(lightPos.w, dist));
                light += (m_SpotLightColor[i].xyz * intensity);
            }
        }
        light = min(light, 3.5);
    #endif
    
    //Fogcolor calculation
    fog.xyz *= (max(inColor.r, inColor.w) * m_Sunlight.w);

    #ifdef ENVIRONMENTMAPPING
        vec3 I = normalize(g_CameraPosition - worldpos).xyz;
        vec3 N = normalize((g_WorldMatrix * vec4(inNormal.xyz, 0.0)).xyz);

        envRef.xyz = reflect(I, N);
        envRef.w = (m_Fresnel.x + m_Fresnel.y * pow(1.0 + dot(I, N), m_Fresnel.z)) * inTexCoord2.z;
    #endif
}