////////////////////////////////////////////////////////////////////////////////////
/////    WaterFilter Shader for post water effects (depth)                     /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#else
    uniform vec2 g_FrustumNearFar;
#endif

uniform mat4 g_ViewProjectionMatrixInverse;

uniform sampler2D m_Texture;
uniform sampler2D m_DepthTexture;
uniform sampler2D m_WaterPassTexture;
uniform sampler2D m_WaterPassDepthTexture;

uniform bool m_Underwater;
uniform float m_WaterDepth;
uniform float m_UnderwaterFogDensity;
uniform vec3 m_UnderwaterColor;

varying vec2 texCoord;

const vec4 water = vec4(0.1843, 0.2588, 0.4471, 1.0);

//Depth texture to world position
vec3 getPosition(in float depth, in vec2 uv){
    vec4 pos = vec4(uv, depth, 1.0) * 2.0 - 1.0;
    pos = g_ViewProjectionMatrixInverse * pos;
    return pos.xyz / pos.w;
}

void main(){
    vec4 fragColor = texture2D(m_Texture, texCoord);
    vec4 waterColor = texture2D(m_WaterPassTexture, texCoord);
    
    if(!m_Underwater){
        float depth = texture2D(m_DepthTexture, texCoord).r;
        float waterDepth = texture2D(m_WaterPassDepthTexture, texCoord).r;
        
        #ifdef LOGARITHMIC_DEPTH_BUFFER
            depth = convertLogDepthBufferToZ(depth, g_FrustumNearFar.x, g_FrustumNearFar.y);
            waterDepth = convertLogDepthBufferToZ(waterDepth, g_FrustumNearFar.x, g_FrustumNearFar.y);
        #endif
        
        vec3 posWorld = getPosition(depth, texCoord);
        vec3 posWater = getPosition(waterDepth, texCoord);
        
        float ydiff = (posWater.y - posWorld.y);
        float dist = mix((posWater.y - posWorld.y), distance(posWater, posWorld), 0.5);  //TODO
        float fade = smoothstep(0.0, m_WaterDepth, dist) * 0.9 + 0.1;
        fade *= smoothstep(0.0, 0.3, ydiff);
        if(posWater.y < posWorld.y){
            fade = 0.75;
        }
        fragColor = mix(fragColor, waterColor, fade);
    }
    else{
        float depth = texture2D(m_WaterPassDepthTexture, texCoord).r;
        
        #ifdef LOGARITHMIC_DEPTH_BUFFER
            depth = convertLogDepthBufferToZ(depth, g_FrustumNearFar.x, g_FrustumNearFar.y);
        #endif

        float dist = (2.0 * g_FrustumNearFar.x) / (g_FrustumNearFar.y + g_FrustumNearFar.x - depth * (g_FrustumNearFar.y - g_FrustumNearFar.x));
        float fog = exp2(-m_UnderwaterFogDensity * m_UnderwaterFogDensity * dist * dist * 1.442695);
        fog = clamp(fog, 0.0, 1.0);
        fragColor = mix(fragColor, waterColor, 0.9);
        fragColor.rgb = mix(m_UnderwaterColor, fragColor.rgb, fog);
    }
    
    gl_FragColor = fragColor;
}
