////////////////////////////////////////////////////////////////////////////////////
/////    Voxel-Terrain Shader for use with marching-cubes generated Terrain    /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#extension GL_EXT_gpu_shader4 : enable
#extension GL_EXT_texture_array : enable

#import "Shaders/Lib/Selector.glsllib"

#ifndef VERTEXLIGHTING
    #import "Shaders/Lib/Spotlighting.glsllib"
#endif

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

uniform vec3 g_CameraDirection;

uniform sampler2DArray m_Textures_Array;
uniform sampler2DArray m_Textures_Array_Ores;
uniform sampler2DArray m_NormalMaps_Array;
uniform sampler2D m_CrackedTexture;
uniform bool m_PickingActive;
uniform vec4 m_PickingPosition;

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 light;
varying vec3 blending;
varying vec4 texCoord2;
varying vec4 texCoord3;
varying vec4 texCoord4;
varying vec4 texCoord5;
varying vec4 texCoord6;
varying vec4 texCoord7;
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const float shininess = 1.0;

vec4 getTriPlanarBlend(in sampler2DArray array, in vec4 coords){
    vec3 texCoord = vec3(coords.xyw);
    texCoord.xy = coords.zy * 0.25;
    vec4 col1 = texture2DArray(array, texCoord);
    texCoord.xy = coords.xz * 0.25;
    vec4 col2 = texture2DArray(array, texCoord);
    texCoord.xy = coords.xy * 0.25;
    vec4 col3 = texture2DArray(array, texCoord);
    vec4 tex = col1 * blending.x + col2 * blending.y + col3 * blending.z;
    return tex;
}

vec4 getTriPlanarBlend(in sampler2D texture, in float scale, in vec3 coords){
    vec2 texCoord = vec2(coords.xy);
    texCoord.xy = coords.zy * 0.75 * scale;
    vec4 col1 = texture2D(texture, texCoord);
    texCoord.xy = coords.xz * 0.75 * scale;
    vec4 col2 = texture2D(texture, texCoord);
    texCoord.xy = coords.xy * 0.75 * scale;
    vec4 col3 = texture2D(texture, texCoord);
    vec4 tex = col1 * blending.x + col2 * blending.y + col3 * blending.z;
    return tex;
}

vec4 calculateTriPlanarDiffuseBlend(in vec3 position){
    //Texturecoordinates
    vec4 coords = vec4(position, 0.0);

    //FragmentColor
    vec4 fragColor = vec4(0.0, 0.0, 0.0, 1.0);

    ////////////////////////////////////////////////////////////////////////////////////////
    /////             Set1: R=Dirt(1), G=Grass(2), B=Stone(3), 4=Gravel(4)             /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dirt
    if(texCoord2.r > 0.0){
        coords.w = 0.0;
        vec4 dirt = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = dirt * texCoord2.r;
    }
    //Grass
    if(texCoord2.g > 0.0){
        coords.w = 1.0;
        vec4 grass = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, grass, texCoord2.g);
    }
    //Stone
    if(texCoord2.b > 0.0){
        coords.w = 2.0;
        vec4 stone = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, stone, texCoord2.b);
    }
    //Gravel
    if(texCoord2.a > 0.0){
        coords.w = 3.0;
        vec4 gravel = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, gravel, texCoord2.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////            Set2: R=Clay(5), G=Farmland(6), B=Mud(7), 4=Snow(8)               /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Rock
    if(texCoord3.r > 0.0){
        coords.w = 4.0;
        vec4 rock = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, rock, texCoord3.r);
    }
    //Farmland
    if(texCoord3.g > 0.0){
        coords.w = 5.0;
        vec4 forest = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, forest, texCoord3.g);
    }
    //Mud
    if(texCoord3.b > 0.0){
        coords.w = 6.0;
        vec4 mud = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, mud, texCoord3.b);
    }
    //Snow
    if(texCoord3.a > 0.0){
        coords.w = 7.0;
        vec4 snow = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, snow, texCoord3.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////     Set3: R=Sand(9), G=Desertdirt(10), B=Desertstone(11), 4=Solidrock(12)    /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Sand
    if(texCoord4.r > 0.0){
        coords.w = 8.0;
        vec4 sand = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, sand, texCoord4.r);
    }
    //Desertdirt
    if(texCoord4.g > 0.0){
        coords.w = 9.0;
        vec4 desertdirt = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, desertdirt, texCoord4.g);
    }
    //Desertstone
    if(texCoord4.b > 0.0){
        coords.w = 10.0;
        vec4 desertstone = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, desertstone, texCoord4.b);
    }
    //Clay
    if(texCoord4.a > 0.0){
        coords.w = 11.0;
        vec4 clay = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, clay, texCoord4.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////   Set4: R=Dungeonwall(13), G=Dungeonfloor(14), B=Coal(15), A=Hellstone(16)   /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dungeonwall
    if(texCoord5.r > 0.0){
        coords.w = 12.0;
        vec4 dungeonwall = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, dungeonwall, texCoord5.r);
    }
    //Dungeonfloor
    if(texCoord5.g > 0.0){
        coords.w = 13.0;
        vec4 dungeonfloor = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, dungeonfloor, texCoord5.g);
    }
    //Coal
    if(texCoord5.b > 0.0){
        coords.w = 14.0;
        vec4 coal = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, coal, texCoord5.b);
    }
    //Hellstone
    if(texCoord5.a > 0.0){
        coords.w = 15.0;
        vec4 hellstone = getTriPlanarBlend(m_Textures_Array, coords);
        fragColor = mix(fragColor, hellstone, texCoord5.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////       Set5: R=Ironore(0), G=Copper(2), B=Aluminium(3), A=Silver(4)       /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Ironore
    if(texCoord6.r > 0.0){
        coords.w = 0.0;
        vec4 ironore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, ironore, ironore.a * texCoord6.r);
    }
    //Copperore
    if(texCoord6.g > 0.0){
        coords.w = 1.0;
        vec4 copperore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, copperore, copperore.a * texCoord6.g);
    }
    //Aluminiumore
    if(texCoord6.b > 0.0){
        coords.w = 2.0;
        vec4 aluminiumore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, aluminiumore, aluminiumore.a * texCoord6.b);
    }
    //Silverore
    if(texCoord6.a > 0.0){
        coords.w = 3.0;
        vec4 silverore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, silverore, silverore.a * texCoord6.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////       Set6: R=Goldore(5), G=Tungsten(6), B=Cobalt(7), A=Mithril(8)       /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Goldore
    if(texCoord7.r > 0.0){
        coords.w = 4.0;
        vec4 goldore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, goldore, goldore.a * texCoord7.r);
    }
    //Tungstenore
    if(texCoord7.g > 0.0){
        coords.w = 5.0;
        vec4 tungstenore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, tungstenore, tungstenore.a * texCoord7.g);
    }
    //Cobaltore
    if(texCoord7.b > 0.0){
        coords.w = 6.0;
        vec4 cobaltore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, cobaltore, cobaltore.a * texCoord7.b);
    }
    //Mithrilore
    if(texCoord7.a > 0.0){
        coords.w = 7.0;
        vec4 mithrilore = getTriPlanarBlend(m_Textures_Array_Ores, coords);
        fragColor = mix(fragColor, mithrilore, mithrilore.a * texCoord7.a);
    }

    return fragColor;
}

void main(){
    #ifdef TEXTURES
        vec4 fragColor = calculateTriPlanarDiffuseBlend(position);
        fragColor.a = 1.0;
    #else
        vec4 fragColor = vec4(1.0);
    #endif

    //Picking
    if(m_PickingActive){
        vec4 crackedColor = getTriPlanarBlend(m_CrackedTexture, 1.4, position);
        float dist = distance(worldpos.xyz, m_PickingPosition.xyz);
        float intensity = 1.0-smoothstep(m_PickingPosition.w-0.01, m_PickingPosition.w, dist);
        fragColor.rgb = mix(fragColor.rgb, crackedColor.rgb, max(crackedColor.a * intensity - 0.1, 0.0));
    }
    
    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        fragColor.rgb *= calculateSpotlights(light, worldpos, normal, g_CameraDirection, shininess);
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.rgb = mix(fog.rgb, fragColor.rgb, fog.w);
    #endif

    #ifdef SELECTOR
        calculateSelector(worldpos, fragColor);
    #endif

    #ifdef CHUNKBORDER
        if(position.x <= 0.05 || position.x >= 47.95 || position.z <= 0.05 || position.z >= 47.95){
            fragColor.x = 1.0;
        }
        if(position.y <= 0.05 || position.y >= 191.95){
            fragColor.y = 1.0;
        }
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        gl_FragDepth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif
}