////////////////////////////////////////////////////////////////////////////////////
/////    SSAOFilter Shader (Post Processing)                                   /////
/////    Based on JMonkey/ShaderBlow SSAO Filter                               /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#else
    uniform vec2 g_FrustumNearFar;
#endif

uniform vec2 g_Resolution;

uniform sampler2D m_Texture;
uniform sampler2D m_Normals;
uniform sampler2D m_DepthTexture;

uniform vec3 m_FrustumCorner;
uniform float m_SampleRadius;
uniform float m_Intensity;
uniform float m_Scale;
uniform float m_Bias;
uniform float m_FalloffStartDistance;
uniform float m_FalloffRate;

uniform vec3[12] m_Samples;

varying vec2 texCoord;

const float shadowFactor = 0.075;

float getPosition(in vec2 uv, out vec3 position){
    float depthv = texture2D(m_DepthTexture, uv).r;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        depthv = convertLogDepthBufferToZ(depthv, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif

    float depth = (2.0 * g_FrustumNearFar.x) / (g_FrustumNearFar.y + g_FrustumNearFar.x - depthv * (g_FrustumNearFar.y - g_FrustumNearFar.x));
    float x = mix(-m_FrustumCorner.x, m_FrustumCorner.x, uv.x);
    float y = mix(-m_FrustumCorner.y, m_FrustumCorner.y, uv.y);
    position = vec3(x, y, m_FrustumCorner.z) * depth;
    return depthv;
}

vec3 getRandom(in vec2 uv){
    float rand = (fract(uv.x * (g_Resolution.x / 2.0)) * 0.25) + (fract(uv.y * (g_Resolution.y / 2.0)) * 0.5);
    return normalize(vec3(rand));
}

float calculateAmbientOcclusion(in vec2 tc, in vec3 pos, in vec3 norm){
    vec3 diff = vec3(0.0);
    getPosition(tc, diff);
    diff -= pos;
    vec3 v = normalize(diff);
    float d = length(diff) * m_Scale;
    return step(0.00002, d) * max(0.0, dot(norm, v) - m_Bias) * (1.0 / (1.0 + d)) * (m_Intensity + shadowFactor) * smoothstep(0.00002, 0.0027, d);
}

vec3 reflection(in vec3 v1, in vec3 v2){
    vec3 result = 2.0 * dot(v2, v1) * v2;
    result = v1 - result;
    return result;
}

void main(){
    vec3 position = vec3(0.0);
    vec4 normal = texture2D(m_Normals, texCoord);
    vec3 n_normal = normalize(normal.rgb * 2.0 - 1.0);
    
    float depthv = getPosition(texCoord, position);

    //AO nicht berechnen, wenn Depth == 1.0 betraegt oder NOSSAO aktiv
    if(depthv == 1.0 || (normal.a < 1.0 && normal.a > 0.1)){
        gl_FragColor = vec4(1.0);
        return;
    }
    
    vec3 rand = getRandom(texCoord);
    
    float rad = (m_SampleRadius / position.z) + shadowFactor;
    
    vec2 m_DistanceFrustum = vec2(1.0, m_FalloffStartDistance);
    float depth = (m_DistanceFrustum.x / 4.0) / (m_DistanceFrustum.y - depthv * (m_DistanceFrustum.y));
    
    float falloffFactor = exp2(-m_FalloffRate * m_FalloffRate * depth * depth * 1.442695);
    falloffFactor = clamp(falloffFactor, 0.0, 1.0);

    float result;
    if(falloffFactor < 1.0){
        float ao = 0.0;
        
        for(int j = 0; j < 12; ++j){  //12 == Iterations
            vec3 coord1 = reflection(vec3(m_Samples[j]), rand) * vec3(rad);
            ao += calculateAmbientOcclusion(texCoord + coord1.xy * 0.125, position, n_normal) - shadowFactor;
        }

        ao /= 12.0 * (2.35 - shadowFactor);  //12.0 == Iterations
        result = 1.0 - ao;
        result = mix(result, 1.0, 1.0 - falloffFactor);
    }
    else{
        result = 1.0;
    }
    
    gl_FragColor = vec4(result, result, result, 1.0);
}