////////////////////////////////////////////////////////////////////////////////////
/////    Lighting Shader for light-influenced Objects                          /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#ifdef TEXTUREARRAY
    #extension GL_EXT_gpu_shader4 : enable
    #extension GL_EXT_texture_array : enable

    uniform sampler2DArray m_Textures_Array;
    varying vec3 texCoord;
#else
    uniform sampler2D m_Texture;
    varying vec2 texCoord;
#endif

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

#ifdef MASKTEXTURE
    uniform sampler2D m_MaskTexture;
#endif

#ifdef DISCARD
    uniform float m_AlphaDiscardThreshold;
#endif

#ifndef VERTEXLIGHTING
    #import "Shaders/Lib/Spotlighting.glsllib"
#endif

uniform vec3 g_CameraDirection;
uniform vec4 m_Color;

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 light;
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const float shininess = 1.0;

void main(){
    #ifdef MASKTEXTURE
        vec4 mask = texture2D(m_MaskTexture, texCoord);
        if(mask.a < 0.1){
            discard;
        }
    #endif

    #ifdef TEXTUREARRAY
        vec4 fragColor = texture2DArray(m_Textures_Array, texCoord);
        #ifdef COLOR
            fragColor.rgb = mix(fragColor.rgb, m_Color.rgb, m_Color.a);
        #endif
    #else
        #ifdef TEXTURE
            vec4 fragColor = texture2D(m_Texture, texCoord);
            #ifdef COLOR
                fragColor.rgb = mix(fragColor.rgb, m_Color.rgb, m_Color.a);
            #endif
        #else
            #ifdef COLOR
                vec4 fragColor = m_Color;
            #else
                vec4 fragColor = vec4(1.0);
            #endif
        #endif
    #endif

    #ifdef DISCARD
        if(fragColor.a < m_AlphaDiscardThreshold){
            discard;
        }
    #endif
    
    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        fragColor.rgb *= calculateSpotlights(light, worldpos, normal, g_CameraDirection, shininess);
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.xyz = mix(fog.xyz, fragColor.xyz, fog.w);
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        gl_FragDepth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif
}