////////////////////////////////////////////////////////////////////////////////////
/////    Construction Shader for user-made constructions                       /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#extension GL_EXT_gpu_shader4 : enable
#extension GL_EXT_texture_array : enable

#import "Shaders/Lib/Selector.glsllib"

#ifndef VERTEXLIGHTING
    #import "Shaders/Lib/Spotlighting.glsllib"
#endif

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

uniform vec3 g_CameraDirection;

uniform sampler2DArray m_Textures_Array;
uniform sampler2DArray m_Detail_Textures_Array;
uniform sampler2D m_GridTexture;
uniform sampler2D m_CrackedTexture;
uniform float m_Texture_Scale;
uniform float m_Grid_Scale;
uniform bool m_GridActive;
uniform bool m_PickingActive;
uniform vec4 m_PickingPosition;
uniform vec3 m_PickingBlockPosition;

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 light;
varying vec3 texCoord;
varying vec4 fog;
varying float gridfade;
varying float id;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const float shininess = 64.0;

void main(){

    vec4 fragColor = texture2DArray(m_Textures_Array, texCoord);

    #ifdef DETAILTEXTURES
        vec3 dTexCoord = vec3(texCoord.xy * 5.0, texCoord.z);
        fragColor.xyz *= texture2DArray(m_Detail_Textures_Array, dTexCoord).xyz;
    #endif

    //Picking
    if(m_PickingActive){
        vec2 newTexCoord = vec2(texCoord.xy * 2.0);
        vec4 crackedColor = texture2D(m_CrackedTexture, newTexCoord.xy);
        float dist = distance(worldpos.xyz, m_PickingPosition.xyz);
        float blockdist = distance(worldpos.xyz, m_PickingBlockPosition.xyz);
        float intensity = (1.0-smoothstep(m_PickingPosition.w-0.01, m_PickingPosition.w, dist));
        
        //PickingPosition restriction
        if(position.x < (m_PickingBlockPosition.x-0.001) || position.x > (m_PickingBlockPosition.x+1.001) ||
            position.y < (m_PickingBlockPosition.y-0.001) || position.y > (m_PickingBlockPosition.y+1.001) ||
            position.z < (m_PickingBlockPosition.z-0.001) || position.z > (m_PickingBlockPosition.z+1.001)){
            intensity = 0.0;
        }
        
        fragColor.rgb = mix(fragColor.rgb, crackedColor.rgb, max(crackedColor.a * intensity - 0.1, 0.0));
    }

    //Grid
    if(m_GridActive){
        vec2 gTexCoord = vec2(texCoord.xy * (m_Grid_Scale * 2.0));
        vec4 gridColor = texture2D(m_GridTexture, gTexCoord);
        fragColor.xyz = mix(fragColor.xyz, gridColor.xyz, max(gridColor.a*0.9 - gridfade, 0.0));
    }
    
    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        fragColor.rgb *= calculateSpotlights(light, worldpos, normal, g_CameraDirection, shininess);
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.xyz = mix(fog.xyz, fragColor.xyz, fog.w);
    #endif
    
    #ifdef SELECTOR
        calculateSelector(worldpos, fragColor);
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        float depth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
        gl_FragDepth = depth + (id * 0.000005);  //0.000001
    #endif
}