////////////////////////////////////////////////////////////////////////////////////
/////    Vehicle Shader for vehicles with different paintjobs                  /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#extension GL_EXT_gpu_shader4 : enable
#extension GL_EXT_texture_array : enable

#import "Common/ShaderLib/Optics.glsllib"

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

uniform sampler2DArray m_Textures_Array;

#ifndef VERTEXLIGHTING
    #ifndef NUMSPOTLIGHTS
        #define NUMSPOTLIGHTS 1
    #endif
    uniform vec4 m_SpotLightColor[NUMSPOTLIGHTS]; //RGB, A=MaxLightRadius (frontal)
    uniform vec4 m_SpotLightPosition[NUMSPOTLIGHTS]; //XYZ, W=Range (raycasted)
    uniform vec4 m_SpotLightDirection[NUMSPOTLIGHTS]; //XYZ, W=OuterRadius
#endif

#ifdef ENVIRONMENTMAPPING
    uniform ENVMAP m_EnvironmentMap;
    varying vec4 envRef;
#endif

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 light;
varying vec2 texCoord;
varying vec3 texCoord2;  //X=TextureLayer, Y=Transparency (only relevant for transparent parts), Z=ReflectionAmount (1=default, 0=off)
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

void main(){
    #ifdef TEXTURES
        vec3 newTexCoord = vec3(texCoord.x, texCoord.y, texCoord2.x);
        vec4 fragColor = texture2DArray(m_Textures_Array, newTexCoord);
    #else
        vec4 fragColor = vec4(1.0, 0.0, 0.0, 1.0);
    #endif

    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        vec3 lighting = vec3(light);
        for(int i = 0; i < NUMSPOTLIGHTS; i++){
            if(m_SpotLightColor[i].w > 0.0){
                vec4 lightPos = m_SpotLightPosition[i];
                vec4 lightDir = m_SpotLightDirection[i];
                vec4 dir = vec4((worldpos.xyz - lightPos.xyz), 0.0);
                float dist = distance(worldpos.xyz, lightPos.xyz);
                float spotdot = dot(normalize(lightDir.xyz), normalize(dir.xyz));
                float intensity = smoothstep(1.0 - lightDir.w, 1.0, spotdot) * (1.0 - smoothstep(m_SpotLightColor[i].w * 0.5, m_SpotLightColor[i].w, dist)) * clamp((dot(normal, normalize(lightDir.xyz * -1.0)) + 0.25), 0.0, 1.0);
                intensity *= (1.0 - step(lightPos.w, dist));
                lighting += (m_SpotLightColor[i].xyz * intensity);
            }
        }
        lighting = min(lighting, 3.5);
        fragColor.rgb *= lighting;
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.rgb = mix(fog.rgb, fragColor.rgb, fog.w);
    #endif

    #ifdef TRANSPARENCY
        fragColor.a = texCoord2.y;
    #endif

    #ifdef ENVIRONMENTMAPPING
        vec4 envColor = Optics_GetEnvColor(m_EnvironmentMap, envRef.xyz);
        #ifndef VERTEXLIGHTING
            envColor.rgb *= lighting;
        #else
            envColor.rgb *= light;
        #endif
        fragColor.rgb = mix(fragColor.rgb, envColor.rgb, envRef.w);
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        gl_FragDepth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif
}