////////////////////////////////////////////////////////////////////////////////////
/////    Ore Shader for use with marching-cubes generated Ores                 /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#extension GL_EXT_gpu_shader4 : enable
#extension GL_EXT_texture_array : enable

#import "Shaders/Lib/Selector.glsllib"

#ifndef VERTEXLIGHTING
    #import "Shaders/Lib/Spotlighting.glsllib"
#endif

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

uniform vec3 g_CameraDirection;

uniform sampler2DArray m_Textures_Array;
uniform sampler2DArray m_NormalMaps_Array;
uniform sampler2D m_CrackedTexture;
uniform float m_Texture_Scale;
uniform bool m_PickingActive;
uniform vec4 m_PickingPosition;

varying vec3 position;
varying vec3 texCoord;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 blending;
varying vec3 light;
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const float shininess = 1.0;

vec4 getTriPlanarBlend(in vec4 coords, in vec3 blending){
    vec3 texCoord = vec3(coords.xyw);
    texCoord.xy = coords.zy * m_Texture_Scale;
    vec4 col1 = texture2DArray(m_Textures_Array, texCoord);
    texCoord.xy = coords.xz * m_Texture_Scale;
    vec4 col2 = texture2DArray(m_Textures_Array, texCoord);
    texCoord.xy = coords.xy * m_Texture_Scale;
    vec4 col3 = texture2DArray(m_Textures_Array, texCoord);
    vec4 tex = col1 * blending.x + col2 * blending.y + col3 * blending.z;
    return tex;
}

vec4 getTriPlanarBlend(in sampler2D texture, in vec3 coords, in vec3 blending){
    vec2 texCoord = vec2(coords.xy);
    texCoord.xy = coords.zy * (m_Texture_Scale * 3.0);
    vec4 col1 = texture2D(texture, texCoord);
    texCoord.xy = coords.xz * (m_Texture_Scale * 3.0);
    vec4 col2 = texture2D(texture, texCoord);
    texCoord.xy = coords.xy * (m_Texture_Scale * 3.0);
    vec4 col3 = texture2D(texture, texCoord);
    vec4 tex = col1 * blending.x + col2 * blending.y + col3 * blending.z;
    return tex;
}

vec4 calculateTriPlanarDiffuseBlend(in vec3 position, in vec3 blending){
    //Texturecoordinates
    vec4 coords = vec4(position, texCoord.z);
    
    //FragmentColor
    vec4 fragColor = getTriPlanarBlend(coords, blending);
    return fragColor;
}

void main(){
    #ifdef TEXTURES
        vec4 fragColor = calculateTriPlanarDiffuseBlend(position, blending);
    #else
        vec4 fragColor = vec4(1.0);
    #endif
    
    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        fragColor.rgb *= calculateSpotlights(light, worldpos, normal, g_CameraDirection, shininess);
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.rgb = mix(fog.rgb, fragColor.rgb, fog.w);
    #endif
    
    #ifdef SELECTOR
        calculateSelector(worldpos, fragColor);
    #endif

    //Picking
    if(m_PickingActive){
        vec4 crackedColor = getTriPlanarBlend(m_CrackedTexture, position, blending);
        float dist = distance(worldpos.xyz, m_PickingPosition.xyz);
        float intensity = 1.0-step(m_PickingPosition.w, dist);
        fragColor.rgb = mix(fragColor.rgb, crackedColor.rgb, max(crackedColor.a * intensity - 0.1, 0.0));
    }

    #ifdef CHUNKBORDER
        if(position.x <= 0.05 || position.x >= 47.95 || position.z <= 0.05 || position.z >= 47.95){
            fragColor.x = 1.0;
        }
        if(position.y <= 0.05 || position.y >= 191.95){
            fragColor.y = 1.0;
        }
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        gl_FragDepth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif
}