////////////////////////////////////////////////////////////////////////////////////
/////    LOD Shader for combined LOD-Terrain, -Vegetation and -Objects         /////
/////    Author: Danny Imlau (JIW-Games)                                       /////
/////    Fragment-Shader                                                       /////
////////////////////////////////////////////////////////////////////////////////////

#extension GL_EXT_gpu_shader4 : enable
#extension GL_EXT_texture_array : enable

#import "Shaders/Lib/Selector.glsllib"

#ifndef VERTEXLIGHTING
    #import "Shaders/Lib/Spotlighting.glsllib"
#endif

#ifdef LOGARITHMIC_DEPTH_BUFFER
    #import "Shaders/Lib/LogarithmicDepth.glsllib"
#endif

uniform vec3 g_CameraDirection;

uniform sampler2DArray m_Textures_Array_Terrain;
uniform sampler2DArray m_Textures_Array_Constructions;
uniform sampler2DArray m_Textures_Array_Vegetation;
uniform sampler2DArray m_Textures_Array_Snow;
uniform sampler2DArray m_Textures_Array_Objects;

varying vec3 position;
varying vec3 worldpos;
varying vec3 normal;
varying vec3 blending;
varying vec3 light;
varying vec4 texCoord;
varying vec4 texCoord2;
varying vec4 texCoord3;
varying vec4 texCoord4;
varying vec4 texCoord5;
varying vec2 texCoord8;
varying vec4 fog;

#ifdef LOGARITHMIC_DEPTH_BUFFER
    varying vec4 positionProjectionSpace;
#endif

const float shininess = 1.0;

vec4 getTriPlanarBlend(in vec4 coords){
    vec3 texCoord = vec3(coords.xyw);
    texCoord.xy = coords.zy * 0.1875;  //0.25
    vec4 col1 = texture2DArray(m_Textures_Array_Terrain, texCoord);
    texCoord.xy = coords.xz * 0.1875;
    vec4 col2 = texture2DArray(m_Textures_Array_Terrain, texCoord);
    texCoord.xy = coords.xy * 0.1875;
    vec4 col3 = texture2DArray(m_Textures_Array_Terrain, texCoord);
    vec4 tex = col1 * blending.x + col2 * blending.y + col3 * blending.z;
    return tex;
}

vec4 calculateTriPlanarDiffuseBlend(in vec3 position){
    //Texturecoordinates
    vec4 coords = vec4(position, 0.0);

    //FragmentColor
    vec4 fragColor = vec4(0.0, 0.0, 0.0, 1.0);

    ////////////////////////////////////////////////////////////////////////////////////////
    /////             Set1: R=Dirt(1), G=Grass(2), B=Stone(3), 4=Gravel(4)             /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dirt
    if(texCoord2.r > 0.0){
        coords.w = 0.0;
        vec4 dirt = getTriPlanarBlend(coords);
        fragColor = dirt * texCoord2.r;
    }
    //Grass
    if(texCoord2.g > 0.0){
        coords.w = 1.0;
        vec4 grass = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, grass, texCoord2.g);
    }
    //Stone
    if(texCoord2.b > 0.0){
        coords.w = 2.0;
        vec4 stone = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, stone, texCoord2.b);
    }
    //Gravel
    if(texCoord2.a > 0.0){
        coords.w = 3.0;
        vec4 gravel = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, gravel, texCoord2.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////              Set2: R=Clay(5), G=Forest(6), B=Mud(7), 4=Snow(8)               /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Rock
    if(texCoord3.r > 0.0){
        coords.w = 4.0;
        vec4 rock = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, rock, texCoord3.r);
    }
    //Forest
    if(texCoord3.g > 0.0){
        coords.w = 5.0;
        vec4 forest = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, forest, texCoord3.g);
    }
    //Mud
    if(texCoord3.b > 0.0){
        coords.w = 6.0;
        vec4 mud = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, mud, texCoord3.b);
    }
    //Snow
    if(texCoord3.a > 0.0){
        coords.w = 7.0;
        vec4 snow = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, snow, texCoord3.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////     Set3: R=Sand(9), G=Desertdirt(10), B=Desertstone(11), 4=Solidrock(12)    /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Sand
    if(texCoord4.r > 0.0){
        coords.w = 8.0;
        vec4 sand = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, sand, texCoord4.r);
    }
    //Desertdirt
    if(texCoord4.g > 0.0){
        coords.w = 9.0;
        vec4 desertdirt = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, desertdirt, texCoord4.g);
    }
    //Desertstone
    if(texCoord4.b > 0.0){
        coords.w = 10.0;
        vec4 desertstone = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, desertstone, texCoord4.b);
    }
    //Clay
    if(texCoord4.a > 0.0){
        coords.w = 11.0;
        vec4 clay = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, clay, texCoord4.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////      Set4: R=Dungeonwall(13), G=Dungeonfloor(14), B=Bonewall(15)             /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dungeonwall
    if(texCoord5.r > 0.0){
        coords.w = 12.0;
        vec4 dungeonwall = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, dungeonwall, texCoord5.r);
    }
    //Dungeonfloor
    if(texCoord5.g > 0.0){
        coords.w = 13.0;
        vec4 dungeonfloor = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, dungeonfloor, texCoord5.g);
    }
    //Bonewall
    if(texCoord5.b > 0.0){
        coords.w = 14.0;
        vec4 bonewall = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, bonewall, texCoord5.b);
    }
    //Hellstone
    if(texCoord5.a > 0.0){
        coords.w = 15.0;
        vec4 hellstone = getTriPlanarBlend(coords);
        fragColor = mix(fragColor, hellstone, texCoord5.a);
    }

    return fragColor;
}

vec4 calculateDiffuseBlend(in vec4 texCoords){
    //FragmentColor
    vec4 fragColor = vec4(0.0);

    //FragmentColor
    vec3 coords = vec3(texCoord.xy * 4.0, 0.0);

    ////////////////////////////////////////////////////////////////////////////////////////
    /////             Set1: R=Dirt(1), G=Grass(2), B=Stone(3), 4=Gravel(4)             /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dirt
    if(texCoord2.r > 0.0){
        coords.z = 0.0;
        vec4 dirt = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = dirt * texCoord2.r;
    }
    //Grass
    if(texCoord2.g > 0.0){
        coords.z = 1.0;
        vec4 grass = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, grass, texCoord2.g);
    }
    //Stone
    if(texCoord2.b > 0.0){
        coords.z = 2.0;
        vec4 stone = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, stone, texCoord2.b);
    }
    //Gravel
    if(texCoord2.a > 0.0){
        coords.z = 3.0;
        vec4 gravel = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, gravel, texCoord2.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////              Set2: R=Clay(5), G=Forest(6), B=Mud(7), 4=Snow(8)               /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Clay
    if(texCoord3.r > 0.0){
        coords.z = 4.0;
        vec4 clay = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, clay, texCoord3.r);
    }
    //Forest
    if(texCoord3.g > 0.0){
        coords.z = 5.0;
        vec4 forest = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, forest, texCoord3.g);
    }
    //Mud
    if(texCoord3.b > 0.0){
        coords.z = 6.0;
        vec4 mud = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, mud, texCoord3.b);
    }
    //Snow
    if(texCoord3.a > 0.0){
        coords.z = 7.0;
        vec4 snow = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, snow, texCoord3.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////     Set3: R=Sand(9), G=Desertdirt(10), B=Desertstone(11), 4=Solidrock(12)    /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Sand
    if(texCoord4.r > 0.0){
        coords.z = 8.0;
        vec4 sand = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, sand, texCoord4.r);
    }
    //Desertdirt
    if(texCoord4.g > 0.0){
        coords.z = 9.0;
        vec4 desertdirt = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, desertdirt, texCoord4.g);
    }
    //Desertstone
    if(texCoord4.b > 0.0){
        coords.z = 10.0;
        vec4 desertstone = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, desertstone, texCoord4.b);
    }
    //Solidrock
    if(texCoord4.a > 0.0){
        coords.z = 11.0;
        vec4 solidrock = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, solidrock, texCoord4.a);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    /////      Set4: R=Dungeonwall(13), G=Dungeonfloor(14), B=Bonewall(15)             /////
    ////////////////////////////////////////////////////////////////////////////////////////
    //Dungeonwall
    if(texCoord5.r > 0.0){
        coords.z = 12.0;
        vec4 dungeonwall = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, dungeonwall, texCoord5.r);
    }
    //Dungeonfloor
    if(texCoord5.g > 0.0){
        coords.z = 13.0;
        vec4 dungeonfloor = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, dungeonfloor, texCoord5.g);
    }
    //Bonewall
    if(texCoord5.b > 0.0){
        coords.z = 14.0;
        vec4 bonewall = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, bonewall, texCoord5.b);
    }
    //Hellstone
    if(texCoord5.a > 0.0){
        coords.z = 15.0;
        vec4 hellstone = texture2DArray(m_Textures_Array_Terrain, coords);
        fragColor = mix(fragColor, hellstone, texCoord5.a);
    }
    
    return fragColor;
}

void main(){
    vec4 fragColor = vec4(1.0);

    //Since all LOD-Geometries are merged into one, the W-Coordinate of the texCoord is
    //responsible for determination. To avoid problems due to inaccuracy, a 0.5 tolerance
    //was added.
    //0=Terrain, 1=Blocks, 2=Vegetation, 3=Objects, 4=Water(discard)
    if(texCoord.w <= 0.5){
        #ifdef TRIPLANARBLENDING
            fragColor = calculateTriPlanarDiffuseBlend(position);
        #else
            fragColor = calculateDiffuseBlend(texCoord);
        #endif
    }
    else if(texCoord.w <= 2.5){
        vec3 newTexCoord = vec3(texCoord.x, 1.0-texCoord.y, texCoord.z);
        fragColor = texture2DArray(m_Textures_Array_Vegetation, newTexCoord);
        if(fragColor.a <= 0.9){
            discard;
        }

        if(texCoord8.x > 0.5){
            vec4 snow = texture2DArray(m_Textures_Array_Snow, vec3(texCoord.x * 4.0, texCoord.y * 4.0, texCoord8.x - 1.0));
            fragColor.rgb = mix(fragColor.rgb, snow.rgb, snow.a);
        }
    }
    else if(texCoord.w > 3.5){
        discard;
    }
    else{
        vec3 newTexCoord = vec3(texCoord.x, 1.0-texCoord.y, texCoord.z);
        fragColor = texture2DArray(m_Textures_Array_Objects, newTexCoord);
    }
    fragColor.a = 1.0;

    
    //If normal-quality light is enabled: Per-Pixel-Lighting calculation (dynamic Spotlights)
    #ifndef VERTEXLIGHTING
        fragColor.rgb *= calculateSpotlights(light, worldpos, normal, g_CameraDirection, shininess);
    #else
        fragColor.rgb *= light;
    #endif
    
    #ifdef FOG
        fragColor.rgb = mix(fog.rgb, fragColor.rgb, fog.w);
    #endif
    
    #ifdef SELECTOR
        calculateSelector(worldpos, fragColor);
    #endif
    
    gl_FragColor = fragColor;

    #ifdef LOGARITHMIC_DEPTH_BUFFER
        gl_FragDepth = computeLogDepthBuffer(positionProjectionSpace.z, g_FrustumNearFar.x, g_FrustumNearFar.y);
    #endif
}