/****************************
	B a n k I n t f  -  A Java interface for Rising World banking plug-ins.

	GuiDlgBox.java - The base class for the plug-in dialogue boxes..

	Created by : Maurizio M. Gavioli 2016-12-04

	(C) Maurizio M. Gavioli (a.k.a. Miwarre), 2016
	Licensed under the Creative Commons by-sa 3.0 license (see http://creativecommons.org/licenses/by-sa/3.0/ for details)

*****************************/

package com.vistamaresoft.bankintf;

/**
 * An interface defining some common services offered by banks and which
 * banking plug-ins for Rising World are supposed to support.
 * 
 * <p>In each operation, each player is identified by his respective Rising
 * World DB ID (as returned by the Player.getDbId() method); DB ID's are
 * guaranteed to be persistent and unique, more than player names.
 * <p>The interface has the concept of two special accounts:
 * *) The WORLD account, which is intended as the source of money other plug-ins
 * 		grant to or withdraw from players for any reason (time spent logged on,
 * 		activities done, etc.)
 * *) The BANK account, which is the source of money the bank gives to or
 * 		withdraws from players for any reason (interest, mortgage, etc...)
 * <p>These two special accounts are also identified by account numbers which
 * cannot occur as player DB ID's.
 * <p>Only banking plug-ins are expected to use the BANK account and only one
 * banking plug-in is expected to run at a time.
 * <p>All plug-ins are free to use the WORLD account, but it is suggested that
 * an appropriate <i>reason</i> parameter is provided for all operations which
 * support it, to ensure that bank statements make sense to the players.
 * <p>By default, the WORLD and the BANK accounts have unlimited cash, but a
 * money system is free to implement its own restrictions to the money flow.
 * <p><b>Setup</b>:
 * <p>For <b>consumer plug-ins</b> (i.e. plug-ins which only use services
 * provided by banks), it is enough to add the <i>BankIntf.jar</i> to the Java
 * project of your plug-in (the details depends upon the specific IDE you are
 * using); the <i>BankIntf.jar</i> <b>must not</b> be included within your plug-in or
 * shipped with it: it is just for the Java compiler to know the details of the
 * interface at compile time.
 * <p>For <b>provider plug-ins</b> (i.e. plug-ins which provide banking
 * services), the compiled <i>BankIntf.class</i> shall be included in the final
 * .jar of your plug-in, within its proper hierarchy of package directories,
 * parallel to the package of your plug-in.
 * <p>Again, how to achieve this depends on the specific IDE you are using; the
 * simpler set-up I found is to add the whole hierarchy of directories of the
 * interface source code (i.e. com/vistamaresoft/bankintf/BankIntf.java</i>)
 * in your project, parallel to your plug-in package (i.e. usually, directly
 * under <i>src</i>).
 * <p>Adding the <i>BankIntf.jar</i> as it is within the <i>.jar</i> of your
 * plug-in is not going to work, as Rising World will not find it. 
 * <p>A plug-in which is both a provider and a consumer, will need both set-up
 * steps described above.
 */
public interface BankIntf
{
	// Return value codes
	/**	Operation has been successful. */
	public static final		int		ERR_SUCCESS				= 0;
	/**	Operation is not supported. */
	public static final		int		ERR_NOT_SUPPORTED		= -1;
	/**	Operation is not allowed. */
	public static final		int		ERR_DENY				= -2;
	/**	The source account does not exist. */
	public static final		int		ERR_FROM_ACCOUNT_UNKNOWN= -3;
	/**	The destination account does not exist. */
	public static final		int		ERR_TO_ACCOUNT_UNKNOWN	= -4;
	/**	The source account does not have enough money for the operation. */
	public static final		int		ERR_NOT_ENOUGH_MONEY	= -5;
	// Special account codes
	/**	The 'account number' of the Bank. */
	public static final		int		BANK_ID					= 0;
	/**	The 'account number' of the 'World'. */
	public static final		int		WORLD_ID				= -1;

	/**
		Returns the currency name appropriate for the number
		(to account for singular and plural forms).

		<p>The <b>name</b> of the currency is distinct from its <b>symbol</b>:
		the former is to be used in textual contexts, while the latter is for
		numeric or tabular contexts.

		<p>It is up to the implementation to decide how many forms to support:
		just one ("euro"), two ("franc" and "francs") or more.

		@param	number	the number of units for which the name is queried.
		@return	the currency name appropriate for the number of units.
	*/
	public String getCurrencyName(int number);

	/**
		Return the symbol for the currency used.

		<p>The <b>name</b> of the currency is distinct from its <b>symbol</b>:
		the former is to be used in textual contexts, while the latter is for
		numeric or tabular contexts.

		@return	the currency symbol as a String.
	*/
	public String getCurrencySymbol();

	/**
		Checks whether the given account has at least the given amount of money
		available in his account.

		<p>It is a decision of the implementation whether to support money
		reservations for future payments (pre-authorisations) or not.
		If it does support them, the check shall be done on the amount of money
		freely available on the account, once all encumbrances are discounted.

		@param	player_dbid	the DB ID identifying the account.
		@param	amount		the required available amount of money.
		@return	true if at least the given amount of money is available to the account.
	*/
	public boolean hasAvailableMoney(int player_dbid, double amount);

	/**
	 * Returns the account balance for the given account. May be encumbered by
	 * future obligations.

	 * @param	player_dbid	the DB ID of the account.
	 * @return	the account balance or null if player_dbid is not found.
	 */
	public Double getBalance(int player_dbid);

	/**
	 * Returns an array of TransactionData with the transactions relative to
	 * given player_dbid.
	 * <p>Returns transactions involving player_dbid both as a source and a
	 * destination of money; the sign of the amount of each transaction shows
	 * if the amount is an income or an expense from player_dbid perspective.
	 * <p>Both fromTimeStamp and endTimeStamp are according to the <i>Unix
	 * epoch</i> (milliseconds from January 1st, 1970, 00:00).
	 * 
	 * @param	playerDbId		the account to retrieve transactions for.
	 * @param	fromTimeStamp	the start of the time range to return.
	 * @param	toTimeStamp		the end of the time range to return.
	 * @return	an array of TransactionData, possible empty, on success;
	 * 			null on failure or if the operation is not supported.
	 */
	public default TransactionData[] getTransactions(int playerDbId, long fromTimeStamp,
			long toTimeStamp)
	{	return null; }
	
	/**
	 * Transfers the given amount of money from one account to another.
	 * 
	 * Each of the source and destination account may be one of the two special
	 * accounts WORLD_ID or BANK_ID, to cover the cases of money granted 'by the
	 * system' to a player or collected from him.

	 * @param	from_dbid	the DB ID of the source account.
	 * @param	to_dbid		the DB ID of the destination account.
	 * @param	amount		the amount to transfer.
	 * @param	reason		a human-readable text explaining the reason of the
	 * 						transfer, for the receiving player to read.
	 * @return	ERR_FROM_ACCOUNT_UNKNOWN	if the source account does not exist;
	 * 			ERR_TO_ACCOUNT_UNKNOWN		if the destination account does not exist;
	 * 			ERR_NOT_ENOUGH_MONEY		if the source account does not have enough money;
	 * 			ERR_SUCCESS					on success.
	 */
	public int transferMoney(int from_dbid, int to_dbid, double amount, String reason);

	/**
	 * Reserves the given amount of money from from_dbid in the name of to_dbid
	 * for the given reason.

		<p>The implementation is not required to override this method, if it
		does not support encumbrances.
		<p>The default implementation does nothing and returns ERR_NOT_SUPPORTED.

	 * @param	from_dbid	the DB ID of the account from whom the amount is reserved.
	 * @param	to_dbid		the DB ID of the account in whom name the amount is reserved.
	 * @param	amount		the amount reserved.
	 * @param	reason		a human-readable text explaining the reason of the request,
	 * 						for the paying player to read.
	 * @return	On failure, one of the ERR_ error codes (included ERR_NOT_SUPPORTED);
	 * 			On success, a numeric ID identifying the reserve transaction (for future reference);
	 * 			the numeric ID is guaranteed to be unique among all reserve transactions.
	 */
	public default int reserveMoney(int from_dbid, int to_dbid, double amount, String reason)
	{	return ERR_NOT_SUPPORTED; }

	/**
	 * Requests a player to authorise a payment.

		<p>The implementation is not required to override this method, if it
		does not support pre-payments.
		<p>The default implementation does nothing and returns ERR_NOT_SUPPORTED.
		<p>It is the responsibility of the implementation, if it decides to
		support pre-payments, to present the request to the paying player, give
		him a way to authorise/deny and to inform the recipient of the
		outcome.

	 * @param	from_dbid	the DB ID of the player from whom payment is requested.
	 * @param	to_dbid		the DB ID of the player requesting the payment.
	 * @param	amount		the requested amount.
	 * @param	ref			a numeric ID identifying the request to the caller.
	 * @param	reason		a human-readable text explaining the reason of the request, for
	 * 						the paying player to read.
	 * @return	On failure, one of the ERR_ error codes (included ERR_NOT_SUPPORTED);
	 * 			On success, either ERR_SUCCESS or ERR_DENY, depending on the source player reply.
	 */
	public default int requestPayment(int from_dbid, int to_dbid, double amount, int ref, String reason)
	{	return ERR_NOT_SUPPORTED; }

	/**
	 * Establishes a recurring payment from from_dbid to to_dbid every days days.

		<p>The implementation is not required to override this method, if it
		does not support recurring payments.
		<p>The default implementation does nothing and returns ERR_NOT_SUPPORTED.
		<p>The request is supposed to have been authorised by the paying
		player and the implementation is not required to take any step for this.
		It is however the responsibility of the implementation, if it decides
		to support recurring payments, to charge the paying account at each
		deadline and to notify the paid player if the source account does not
		have enough money to cover the expense.

	 * @param	from_dbid	the DB ID of the paying player.
	 * @param	to_dbid		the DB ID of the paid player.
	 * @param	amount		the amount of each payment.
	 * @param	days		every how many days to pay the amount.
	 * @param	ref			a numeric ID identifying the request to the caller.
	 * @param	reason		a human-readable text explaining the reason of the request, for
	 * 						the paying player to read.
	 * @return	On failure, one of the ERR_ error codes (included ERR_NOT_SUPPORTED);
	 * 			On success, a numeric ID identifying the engagement (for future reference);
	 * 			the numeric ID is guaranteed to be unique among all recurring payment transactions.
	 */
	public default int recurrentPayement(int from_dbid, int to_dbid, double amount, int days, int ref,
			String reason)
	{	return ERR_NOT_SUPPORTED; }

	/**
	 * Stores the data for a money transaction, relative to a certain account.
	 * <p>The dbId and the name fields are relative to the <b>other part</b>
	 * involved in the transaction.
	 */
	public static class TransactionData
	{
		/** The id of the transaction.
		 * <p>This value makes sense only for the providing bank; use 0 when
		 * creating new transactions from scratch.
		 */
		public int		id;
		/** The DB ID of the other part involved in the transaction. */
		public	int		dbId;
		/** The name of the other part involved in the transaction. */
		public	String	name;
		/** The time stamp of the transaction; in milliseconds relative to the
		 * <i>Unix epoch</i>. */
		public	long	timeStamp;
		/** The amount of the transaction. The sign is relative to account this
		 * transaction refers to. */
		public	double	amount;
		/** the reason for the transaction. May be empty, but not null and a
		 * user-readable reason is strongly suggested. */
		public	String	reason;

		/**
		 * Creates a new transaction with the given data. See the descriptions
		 * of the class fields for info about the various parameters. */
		public TransactionData(int id, int dbId, String name, long timeStamp, double amount,
				String reason)
		{
			this.id			= id;
			this.dbId		= dbId;
			this.name		= name;
			this.timeStamp	= timeStamp;
			this.amount		= amount;
			this.reason		= reason;
		}
	}

}
