package net.risingworld.api.example.guestbook;

import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.Threading;
import net.risingworld.api.events.player.PlayerConnectEvent;
import net.risingworld.api.events.player.PlayerElementInteractionEvent;
import net.risingworld.api.events.player.gui.PlayerGuiElementClickEvent;
import net.risingworld.api.events.player.PlayerKeyEvent;
import net.risingworld.api.gui.Font;
import net.risingworld.api.gui.GuiElement;
import net.risingworld.api.gui.GuiImage;
import net.risingworld.api.gui.GuiLabel;
import net.risingworld.api.gui.PivotPosition;
import net.risingworld.api.objects.Player;
import net.risingworld.api.utils.KeyInput;
import net.risingworld.api.worldelements.World3DModel;
import net.risingworld.api.worldelements.WorldElement;

/**
 * This is our event listener. You can either create separate listener for particular
 * events (e.g. a "PlayerListener" for player-related events, a "WorldListener" for
 * world-related events etc), or just put everything into a single class. If desired,
 * you can even put everything into your main plugin class (it just has to implement the
 * interface "Listener" then), although it keeps things more structured if you create
 * a separate class.<br>
 * 
 * All event methods need the annotation "@EventMethod", otherwise they will not
 * be recognized. In addition, you can only use one single parameter for an event
 * method (the event object which is related to the particular event, e.g. 
 * "PlayerCommandEvent" if this is the event method for player commands etc).
 * 
 * @author red51
 */

public class PlayerListener implements Listener{
    
    /** We keep a reference to our main plugin class */
    private Guestbook plugin;
    
    /**
     * The constructor of our class. We pass a reference to our main plugin class
     * as parameter, in order to access some of its variables.
     * @param plugin a reference to our main plugin class (the class which extends
     * "Plugin", in this case it is the "Guestbook" class)
     */
    public PlayerListener(Guestbook plugin){
        this.plugin = plugin;
    }
    
    /**
     * This event is triggered when the player connects / joins the game. This event
     * is useful to "preload" gui elements, world elements etc.
     * @param evt The event object which provides access to the player object (the player
     * who connected)
     */
    @EventMethod(Threading.Sync)
    public void onPlayerConnect(PlayerConnectEvent evt){
        //Get the player object (the player wo just connected)
        Player p = evt.getPlayer();
        
        //If this player joins our game/server for the first time (can be determines
        //with the isNewPlayer() function), we will add him to the guestbook entries
        if(evt.isNewPlayer()){
            plugin.addNewEntry(p.getName(), p.getLastTimeOnline());
        }
        
        //Now we want to load the gui for the player. In order to keep things more
        //structured, we moved the code to a separate method
        initGuiForPlayer(p);
        
        //Finally we want to add all 3d models of the guestbook to the player
        for(World3DModel model : plugin.guestbookModels){
            p.addWorldElement(model);
        }
        
        //We want to listen for the ESC key - so we register it in order to get key events
        p.registerKeys(KeyInput.KEY_ESCAPE);
        p.setListenForKeyInput(true);
    }
    
    /**
     * This event is triggered when the player interacts with the gui, i.e. when
     * he clicks on an element (as long as the "clickable" state of the element is
     * set to true.
     * @param evt The event object which provides access to the player and the gui
     * element the player has clicked
     */
    @EventMethod
    public void onPlayerGuiClick(PlayerGuiElementClickEvent evt){
        //Get the player object
        Player p = evt.getPlayer();
        
        //Get the gui element
        GuiElement element = evt.getGuiElement();
        
        //Now retrieve the "Next" and "Previous" labels from the player attributes
        GuiLabel guestbookPrevious = (GuiLabel)p.getAttribute("gb_previous");
        GuiLabel guestbookNext = (GuiLabel)p.getAttribute("gb_next");
        
        //Check if the clicked gui element is actually the "Next" label...
        if(element == guestbookNext){
            int page = (int)p.getAttribute("gb_page");
            page++;  //Increment (page = page + 1;)
            GuiLabel[] guestbookLabels = (GuiLabel[])p.getAttribute("gb_labels");
            int maxpage = (plugin.entries.size() / guestbookLabels.length);
            if(page > maxpage) page = maxpage;
            setPageForPlayer(p, page);
            
            //Play sound for changing the page
            p.playGameSound("gui_journal_changepage");
        }
        //...or the "Previous" label?
        else if(element == guestbookPrevious){
            int page = (int)p.getAttribute("gb_page");
            page--;  //Decrement (page = page - 1;)
            if(page < 0) page = 0;
            setPageForPlayer(p, page);
            
            //Play sound for changing the page
            p.playGameSound("gui_journal_changepage");
        }
    }
    
    /**
     * This event is triggered when the player interacts with a world element, in our
     * case this might be our guestbook-model. 
     * @param evt the event object provides us access to the player as well as
     * to the according world element (i.e. the world element the player is
     * currently interacting with).
     */
    @EventMethod
    public void onPlayerInteract(PlayerElementInteractionEvent evt){
        //Get the player object
        Player p = evt.getPlayer();
        
        //Get the world element (the player is currently interacting with)
        WorldElement element = evt.getWorldElement();
        
        //We have to check if the element is actually one of our guestbook model, since
        //this event may also be triggered for a different model (maybe even a model
        //from another plugin)
        for(World3DModel model : plugin.guestbookModels){
            if(model == element){
                //Get the current page of this player
                int page = (int)p.getAttribute("gb_page");

                //Retrieve the label-array from the player attributes (we need
                //it in the next step). Remember that we always have to use type-casting
                //when retrieving an attribute
                GuiLabel[] guestbookLabels = (GuiLabel[])p.getAttribute("gb_labels");

                //Now check (just to make sure) if the page is valid (0 <= page <= maxpage)
                int maxpage = (plugin.entries.size() / guestbookLabels.length);
                if(page > maxpage) page = maxpage;

                //Change the page (since this method is needed several times, we moved
                //the code to a separate method
                setPageForPlayer(p, page);

                //Lets play the journal sound (which is quite suitable for this)
                p.playGameSound("gui_journal_show");

                //Show the mouse cursor (necessary in order to allow the player to 
                //interact with the gui)
                p.setMouseCursorVisible(true);

                //Retrieve the guestbook background from the player attributes and 
                //set it to "visible"
                GuiImage guestbookBackground = (GuiImage)p.getAttribute("gb_background");
                guestbookBackground.setVisible(true);

                //As long as the guestbook is visible, we want to disable the ESC
                //key (and use the ESC key for our own purposes, in this case we
                //want to use it to hide the guestbook)
                p.disableClientsideKeys(KeyInput.KEY_ESCAPE);
                
                //We can break the loop here, since we found the element we
                //were looking for
                break;
            }
        }
    }
    
    /**
     * This event is triggered when the player presses a key, at least as long as
     * the particular keys are registered (i.e. the player explicitly listens for their
     * input), if if the "setListenForKeyInput()" method was set to true for the player.
     * @param evt The event object which provides access to the player object and various
     * key input information.
     */
    @EventMethod
    public void onPlayerKeyInput(PlayerKeyEvent evt){
        //Get the player object (the player who pressed the key)
        Player p = evt.getPlayer();
        
        //check if it's the ESC key and if it's actually pressed
        if(evt.isPressed() && evt.getKeyCode() == KeyInput.KEY_ESCAPE){
            //Retrieve the background gui element from the player attributes
            GuiImage guestbookBackground = (GuiImage)p.getAttribute("gb_background");
            
            //If the element is visible, hide it, hide the cursor and re-enable the ESC key
            if(guestbookBackground.isVisible()){
                guestbookBackground.setVisible(false);
                p.enableClientsideKeys(KeyInput.KEY_ESCAPE);
                p.setMouseCursorVisible(false);
            }
        }
    }
    
    
    ////////////////////////////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////
    
    
    private void initGuiForPlayer(Player p){
        //That's the plan: Since most gui elements may vary for different player 
        //(for example, player 1 views page 2 in the guestbook, while player 2 views
        //page 4 at the same time etc), we will create a separate gui element per
        //player.
        //In order to keep a reference to the gui elements, we store them as
        //a player attribute (alternatively we could use a HashMap or something
        //similar, but attributes are more convenient).
        
        //Create image element (our background image). We use relative position coordinates (0.5). The
        //size doesn't matter here, since we want to set proportinal size values.
        GuiImage guestbookBackground = new GuiImage(plugin.guestbookTexture, 0.5f, 0.5f, true, 500, 500, false);
        p.setAttribute("gb_background", guestbookBackground);

        //Set pivot to center
        guestbookBackground.setPivot(PivotPosition.Center);

        //Now we set an absolute height value (700 px), and want a proportinal width
        guestbookBackground.setWidthProportional(1000, false);

        //By default, we set the visibility of the gui element to false, since
        //we only want to show the guestbook when needed
        guestbookBackground.setVisible(false);
        
        //Add the background to the player gui. This has to be done for every
        //gui element you want to show on the player gui.
        p.addGuiElement(guestbookBackground);

        //"Next" and "Previous" buttons for the guestbook. We use the "Handdrawn" font
        //(which is very suitable for a book), set a black font color, and set them
        //"clickable" (i.e. an event will be triggered once the player clicks on 
        //these labels - we use that to change the sites).
        //PS: These labels will be a child of the background element, i.e. if we
        //do any changes to the parent (change visibility, for example), it affects
        //the childs automatically. In addition, the coordinates of these elements
        //are relative to the parents coordinates.
        GuiLabel guestbookPrevious = new GuiLabel("<- Previous", 0.15f, 0.2f, true);
        guestbookPrevious.setFont(Font.Handdrawn);
        guestbookPrevious.setFontColor(0x000000FF);
        guestbookPrevious.setFontSize(34);
        guestbookPrevious.setClickable(true);
        guestbookBackground.addChild(guestbookPrevious);
        p.setAttribute("gb_previous", guestbookPrevious);
        p.addGuiElement(guestbookPrevious);

        GuiLabel guestbookNext = new GuiLabel("Next ->", 0.85f, 0.2f, true);
        guestbookNext.setPivot(PivotPosition.BottomRight);
        guestbookNext.setFont(Font.Handdrawn);
        guestbookNext.setFontColor(0x000000FF);
        guestbookNext.setFontSize(34);
        guestbookNext.setClickable(true);
        guestbookBackground.addChild(guestbookNext);
        p.setAttribute("gb_next", guestbookNext);
        p.addGuiElement(guestbookNext);

        //Next we want to add labels (the individual entries) to the guestbook
        //background element. These labels will be attached as childs as well.
        //Most convenient way is to store all labels in an array. 9 labels on  
        //the left, 9 labels on the right side (so we create an array with a 
        //size of 18).
        
        //Lets begin with the left side:
        GuiLabel[] guestbookLabels = new GuiLabel[18];
        for(int i = 0; i < 9; i++){
            GuiLabel label = new GuiLabel(0.12f, 0.775f - (i * 0.065f), true);
            label.setFont(Font.Handdrawn);
            label.setFontColor(0x000000FF);
            label.setFontSize(30);
            label.setColorCodesEnabled(true);
            guestbookBackground.addChild(label);
            guestbookLabels[i] = label;
            //add label to player gui
            p.addGuiElement(label);
        }
        //Now all labels on the right side:
        for(int i = 0; i < 9; i++){
            GuiLabel label = new GuiLabel(0.55f, 0.775f - (i * 0.065f), true);
            label.setFont(Font.Handdrawn);
            label.setFontColor(0x000000FF);
            label.setFontSize(30);
            label.setColorCodesEnabled(true);
            guestbookBackground.addChild(label);
            guestbookLabels[i + 9] = label;
            //add label to player gui
            p.addGuiElement(label);
        }
        //Remember to add the label-array as attribute to the player
        p.setAttribute("gb_labels", guestbookLabels);
        
        //Another attribute: We want to keep track of the current page of this 
        //player (by default 0 of course)
        p.setAttribute("gb_page", 0);
    }
    
    private void setPageForPlayer(Player player, int page){
        //Store the page in the player attributes
        player.setAttribute("gb_page", page);
            
        //Retrieve the label-array from the player attributes.
        GuiLabel[] guestbookLabels = (GuiLabel[])player.getAttribute("gb_labels");
        
        for(int i = 0; i < guestbookLabels.length && i < plugin.entries.size(); i++){
            int index = page * guestbookLabels.length + i;
            if(index < plugin.entries.size()){
                //Set the text of this entry
                String entry = plugin.entries.get(index);
                guestbookLabels[i].setText(index + ". " + entry);
            }
            else{
                //Otherwise, set an empty text for this label
                guestbookLabels[i].setText("");
            }
        }
    }
    
}
