package net.risingworld.api.example.guestbook;

import java.io.File;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import net.risingworld.api.Plugin;
import net.risingworld.api.database.WorldDatabase;
import net.risingworld.api.utils.CollisionShape;
import net.risingworld.api.utils.Crosshair;
import net.risingworld.api.utils.ImageInformation;
import net.risingworld.api.utils.ModelInformation;
import net.risingworld.api.utils.Utils;
import net.risingworld.api.worldelements.World3DModel;

/**
 * This is our main plugin class. It has to extend "Plugin", and implement the
 * methods "onEnable()" (which is called when the plugin is loaded) and
 * "onDisable()" (which is called when the plugin is unloaded).<br>
 * 
 * It is recommendable to use the "onEnable()" method to load some stuff, and/or
 * register your event listeners etc.
 * 
 * @author red51
 */

public class Guestbook extends Plugin{
    
    //A list containing all our 3d models
    public ArrayList<World3DModel> guestbookModels = new ArrayList<>();
    
    //Texture of our guestbook (background image)
    public ImageInformation guestbookTexture;
    
    //All entries (will be loaded from the world database)
    public ArrayList<String> entries = new ArrayList<>();
    
    @Override
    public void onEnable(){
        initializeGUI();
        initializeModel();
        loadEntriesFromDatabase();
        
        //Create and register our event listener
        registerEventListener(new PlayerListener(this));
        
        //This just prints an output to the console. Very useful for debugging,
        //or just for providing some information for the log
        System.out.println("Guestbook plugin loaded successfully!");
    }

    @Override
    public void onDisable(){
        //Actually there is nothing we want to do when the plugin gets unloaded.
        //So we just keep this method blank.
    }
    
    ////////////////////////////////////////////////////////////////////////////
    
    private void initializeGUI(){
        try{
            //Not very much to do here: Basically we have to decide if we want
            //to create gui elements once and reuse them for every player, or if
            //we want to create separate gui elements per player. Reusable global
            //gui elements are helpful for "static" gui elements, or more precisely,
            //for gui elements which will never change (for example: if you want to
            //put a server logo or server info text to the bottom of the screen).
            //But in our case, we need separate gui elements per player, since the
            //state of the gui elements might be different per player (especially
            //the visibility).
            
            //However, one thing we can still reuse (or better: we should reuse)
            //is the image information object (the reference to the texture, in this
            //case the background image ob the guestbook). We decided to use .dds
            //images (dds images have a much (!) lower memory consumption than 
            //jpg or png images, but of course jpg or png files are also supported).
            //Note: We can use the "getPath()" function to get the file path to
            //our plugin folder. In this case the texture is stored in a subfolder
            //"assets"
            guestbookTexture = new ImageInformation(getPath() + "/assets/guestbook_background.dds");
        }
        catch(Exception e){
            e.printStackTrace();
        }
    }
    
    private void initializeModel(){
        try{
            //Load 3d model (.obj) from pluginfolder/assets (we get the path to our
            //plugin folder by calling "getPath()"
            ModelInformation model = new ModelInformation(getPath() + "/assets/guestbook.obj");
            
            //Load texture (.dds) for 3d model from same folder
            ImageInformation texture = new ImageInformation(getPath() + "/assets/guestbook.dds");
            
            //We want to allow users to create multiple guestbook models in their world.
            //It is also up to the user to provide proper coordinates and rotation
            //values for the models.
            
            //Warning: the next part is just about a convenient way to set coordinates from
            //outside. This is not necessary, it's just to allow users to customize the
            //coordinates easily! 
            //Alternatively you could simply set the coordinates this way:
            //guestbookModel.setPosition(x, y, z);
            //guestbookModel.setRotation(rx, ry, rz);
            
            //But it's much more convenient for server owners to set these values 
            //in a text file, for example. So we simply look for a "coordinates.txt" 
            //file and use the values from there (we accept them in this format:
            //[posx, posy, posz, rotx, roty, rotz] (pos == position, rot == rotation)
            
            //First we have to check if the file even exists, so we create a
            //new "File" object and call the "exists()" function:
            File coordinates = new File(getPath() + "/coordinates.txt");
            if(coordinates.exists()){
                //Now we read the whole string (i.e. the "content") from the file
                String content = Utils.FileUtils.readStringFromFile(coordinates);
                if(content != null && !content.isEmpty()){
                    //We expect one set of coordinates per line.
                    //Easiest way is to split the string into individual lines, and
                    //check every line. This is not the fastest way, but more
                    //convenient
                    String[] lines = content.split("\r\n|\n|\r");  //we have to cover all kinds of line separators
                    
                    //Iterate over every line. For every valid line, we create a
                    //new 3D world element.
                    for(String line : lines){
                        //Optional: We use a regular expression to check if the line contains
                        //a valid string.
                        
                        //NOTE: This is not necessary! Regular expressions can be 
                        //considered as "advanced stuff", so you could simply skip
                        //this part. We just use the regex in order to make sure that
                        //the line matches a certain pattern
                        if(line.matches("\\[([-0-9\\.]+(, |(\\](.)*)$){1}){6}(.)*")){
                            //Get the index of ']' (which is our end of line). If this
                            //method returns -1, this character was not found, and we
                            //just continue to the next line
                            int eol = line.indexOf(']');
                            if(eol > 0){
                                //Every number is separated by a ", ", so we just this
                                //character sequence to split the string (it returns
                                //a string array containing the individual numbers)
                                String[] values = line.substring(1, eol).split(", ");
                                //Only continue if we actually found 6 numbers (i.e. if
                                //the "values" array has a length of 6)
                                if(values.length == 6){
                                    //Create a new 3D world element
                                    World3DModel guestbookModel = create3DModel(model, texture);
                                    
                                    //We double-check if the values are actually numeric.
                                    //Array indices 0-2 are supposed to contain our position coordinates
                                    if(Utils.StringUtils.isNumeric(values[0]) && Utils.StringUtils.isNumeric(values[1]) && Utils.StringUtils.isNumeric(values[2])){
                                        float x = Float.parseFloat(values[0]);
                                        float y = Float.parseFloat(values[1]);
                                        float z = Float.parseFloat(values[2]);
                                        guestbookModel.setPosition(x, y, z);
                                    }
                                    
                                    //Indices 3-5 are supposed to contain our rotation coordinates
                                    if(Utils.StringUtils.isNumeric(values[3]) && Utils.StringUtils.isNumeric(values[4]) && Utils.StringUtils.isNumeric(values[5])){
                                        //People probably provide the rotation values in degree, but
                                        //we need radian angles. So we convert them from degree to
                                        //radian by using one of the math util functions:
                                        float rx = Utils.MathUtils.degreeToRadian(Float.parseFloat(values[3]));
                                        float ry = Utils.MathUtils.degreeToRadian(Float.parseFloat(values[4]));
                                        float rz = Utils.MathUtils.degreeToRadian(Float.parseFloat(values[5]));
                                        
                                        //Our rotation is built from euler rotation angles. World elements
                                        //have a convenient method to convert the euler angles to a proper
                                        //quaternion rotation automatically:
                                        guestbookModel.setRotation(rx, ry, rz);
                                    }
                                    
                                    //Finally we add the new 3D world element to our list of world elements
                                    //(we can access this list once a player connects, and add the models
                                    //to his world)
                                    guestbookModels.add(guestbookModel);
                                }
                            }
                        }
                    }
                }
            }
        }
        catch(Exception e){
            //Here you could add some specific expception handling or loggin. 
            //We simply "print" the whole exception by calling "printStackTrace()"
            e.printStackTrace();
        }
    }
    
    private World3DModel create3DModel(ModelInformation model, ImageInformation texture){
        //We create a new 3D world model. As opposed to our gui elements,
        //the world element never changes, i.e. it is always in the same state
        //for every player. Of course we could use the same approach as for the
        //gui elements (create a new element per player), but this isn't necessary
        //in this case. It is sufficient to create just a single world element,
        //store it in a variable and reuse it for every player
        World3DModel guestbookModel = new World3DModel(model, texture);

        //We want our model to be affected by lights (looks more realistic),
        //so we enable lighting
        guestbookModel.setLightingEnabled(true);

        //Since the player is supposed to interact with the model, we have to
        //set the "interactable" state to true
        guestbookModel.setInteractable(true);

        //We can set a crosshair for this model (it will be displayed when the 
        //player looks at it). The "write" (pen) crosshair might be suitable,
        //but of course we could also choose another one
        guestbookModel.setInteractionCrosshair(Crosshair.Write);

        //We need a collision shape for this model. Otherwise there would be
        //no collision, and the player would be able to "walk through" the model.
        //There are various shapes available, but a "HullCollisionShape" might
        //be a good approach for many models. We could create a "mesh accurate"
        //shape as well, but that would be an "overkill" for our model
        guestbookModel.setCollisionShape(CollisionShape.createHullCollisionShape());

        //We found out that our model is way too big, so we have to scale it down
        guestbookModel.setScale(0.0085f);
        
        return guestbookModel;
    }
    
    private void loadEntriesFromDatabase(){
        //We need access to the world database in order to get all players who have
        //been on this server in the past
        WorldDatabase database = getWorldDatabase();
        
        //We have to use a try-catch block, since getting the result set may throw
        //an SQL exception. We are using "try-with-resources" (since Java 7), since
        //it is always necessary to close a ResultSet once it is no longer needed.
        //If an exception occurs, we may not reach the lower code lines, so we have
        //to make sure that the ResultSet gets closed in any case (either by using
        //a try-finally block, or the more convenient try with resources
        
        //Get all players from the "Player" table in the world database
        try(ResultSet result = database.executeQuery("SELECT * FROM `Player`")){
            //While there is a result available, we read the result data and move
            //the cursor forward one row (this is done in the "result.next()" function)
            while(result.next()){
                //Get the playername (column "Name", String)
                String playername = result.getString("Name");
                
                //Get the timestamp when the player was online for the last time
                //(column "LastTimeOnline", long)
                long date = result.getLong("LastTimeOnline");
                
                //Add a new entry in the guestbook (we created a separate method for
                //that, since it may also be called by our event listener)
                addNewEntry(playername, date);
            }
        }
        catch(SQLException e){
            e.printStackTrace();
        }
    }
    
    public void addNewEntry(String playername, long date){
        //We can use an utils method "getFormattedTimeAndDate()" to convert a
        //timestamp into a readable date/time string
        entries.add(playername + "\n Last visit: " + Utils.GeneralUtils.getFormattedTimeAndDate(date));
    }
    
}
